<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\User;
use App\Model\Payment;
use Illuminate\Support\Facades\Hash;
use DB;
use Illuminate\Database\Eloquent\SoftDeletes;
use PDF;

class UserController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth:web');
    }

    public function index(User $user, Request $request)
    {   
        if (\Auth::user()->type != 'admin') {
            return redirect()->back()->with('error', 'You are not admin!');
        }

        $data['users'] = User::latest()->get();

        if ($request->action == 'pdf') {

            $pdf = PDF::loadView('users.pdf_users', $data, [])->setPaper('a4', 'portrait');

            return $pdf->stream('User List Report. '.date('d-M-Y-').time().'.pdf');
        }else{

            return view('users.index', $data);
        }

    }

    public function edit(User $user)
    {   
        if (\Auth::user()->type != 'admin') {

            $auth_user = Auth::user();
            if ($user->id != $auth_user->id) {
                return redirect()->back()->with('error', 'You are not admin to edit others!');
            }
        }

        $user=$user;
        
        return view('users.edit', compact('user'));
    }

    public function update(User $user, Request $request)
    {
        $auth_user = \Auth::user();
        if ($auth_user->type != 'admin') {

            $auth_user = Auth::user();

            if ($user->id != $auth_user->id) {
                return redirect()->back()->with('error', 'You are not admin to edit others!');
            }
        }

        if (request()->password) {
            $rules = [
                'name' => 'required',
                // 'phone' => 'required',
                'email' => 'required|email|unique:users,email,'.$user->id,
                'password' => 'min:6|confirmed',
                'image' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            ];
        }else{
            $rules = [
                'name' => 'required',
                'email' => 'required|email|unique:users,email,'.$user->id,
                'image' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            ];
        }

        $this->validate(request(), $rules);

        // if(request()->hasFile('image')){
        //     $image = request()->file('image');
        //     $extension = request()->image->getClientOriginalExtension();            
        //     $destinationPath = public_path('/upload/profiles');
        //     $img = Image::make($image->getRealPath());
        //     $img->resize(300, 300, function ($constraint) {
        //         $constraint->aspectRatio();
        //     })->save($destinationPath.'/'.time().'.'.$extension);
        //     $user->image = 'upload/profiles/'.request('name').'-'.time().'.'.$extension;
        // }

        if (request()->hasFile('image')) {
            $image = request()->file('image');
            $name = request('name').'-'.time().'.'.$image->getClientOriginalExtension();
            $destinationPath = public_path('/upload/profiles');
            $image->move($destinationPath, $name);
            
            $user->image = 'upload/profiles/'.$name;
        }

        $user->name = request('name');
        $user->phone = request('phone');
        $user->email = request('email');

        if ($auth_user->type == 'admin' && $auth_user->id != $user->id) {

            // only admin can change actie status of other users
            $user->active = request('active');
        }

        if (request()->password) {
            $user->password = bcrypt(request('password'));
        }

        $user->save();

        if ($auth_user->type == 'admin') {
            
            return redirect()->route('users')->with('success', 'Updated Successfully!');
        }else{
            return redirect()->back()->with('success', 'Updated Successfully!');
        }
    }

    protected function store(Request $request)
    {
        $auth_user = \Auth::user();
        if ($auth_user->type != 'admin') {

            return redirect()->back()->with('error', 'You are not admin to add users!');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:6', 'confirmed'],
        ]);

        User::create([
            // default user type = user
            'type' => 'user',
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('users')->with('success', 'User Created Successfully!');
    }

    public function destroy(User $user)
    {
        $auth_user = \Auth::user();
        if ($auth_user->type != 'admin') {

            return redirect()->back()->with('error', 'You are not admin to delete users!');
        }

        try {

            DB::beginTransaction();

            $user->delete();
            
            DB::commit();

        } catch (\Exception $e) {
            
            DB::rollBack();

            $text="";

            if (strpos($e->getMessage(), 'a foreign key constraint fails') !== false) {
                
                $text = 'This User is already used! It can not be deleted!';
            }
            
            return response()->json(['status'=>'error', 'message'=>'User deleting failed! '.$text]);
        }

        return response()->json(['status'=>'success', 'message'=>'User deleted successfully']);
    }
}