// App controller.
(function ($) {
    $.widget('adsk.appcontroller', {
        options: {
            config: {
                productFullName: '',

                ui: {
                    tabs: {
                        order: [
                            'C','I','S','F'
                        ],
                        tocEnabled: true,
                        indexEnabled: true,
                        favoritesEnabled: true,
                        searchEnabled: true
                    },

                    tools: {
                        order: ['home', 'previous', 'parent', 'following', 'separator',
                                'favorites', 'share', 'separator', 'print'],
                        home: true,
                        previous: true,
                        parent: true,
                        following: true,
                        favorites: true,
                        share: true,
                        print: true
                    },

                    responsive: true
                },

                features: {
                    tabsbarParent: 'banner',
                    banner: true,
                    header: true,
                    toolbar: true,
                    breadcrumbs: true,
                    tabsbar: true,
                    navigation: true,
                    splitter: true
                }
            },
            targetId: 'browser',
            viewControllers: {},
            toc: {},
            defaultTopicData: undefined,
            urlParams: undefined
        },

        /* Private members. */

        _view: undefined,
        _defTopicReady: new $.Deferred(),
        _defTocReady: new $.Deferred(),
        _defContextIdsReady: new $.Deferred(),

        _create: function () {
            console.log('appcontroller: _create');

            var self = this;

            this._setupPolyfills();

            $.when(this._defTopicReady, this._defTocReady, this._defContextIdsReady).done(function() {
                self._addHashchangeHandler();

                // Go to default topic page if there was no hash data specified by the user.
                if (location.hash == '') {
                    var href = self.options.defaultTopicData.attr.href;

                    if (self.options.urlParams !== undefined &&
                        self.options.urlParams.length > 0) {

                        var p = self.options.urlParams;

                        // Process contextId query.
                        if (p['contextId'] !== undefined &&
                            p['contextId'] !== '' ) {

                                // Open specified topic.
                                var id = decodeURIComponent(p['contextId']).toUpperCase(),
                                    contextIdHref = self.options.config.contextIds[id];

                                if (contextIdHref !== undefined && contextIdHref != '') {
                                    href = './'.concat(contextIdHref);
                                }
                        }
                    }

                    // Go to topic page.
                    self._handleOpentopic({}, { href: href });
                } else {
                    $(window).trigger('hashchange');
                }
            });

            this._checkUrlParameters();

            this._createView();

            this._createControllers();

            // Hide main preloader.
            this._view.hidePreloader();

            // Load context id data.
            this._loadContextIdData();

            // Load and process TOC data.
            this._loadAndProcessTOC();
        },

        _setupPolyfills: function() {
            $('html').addClass($.fn.details.support ? 'details' : 'no-details');
        },

        _checkUrlParameters: function() {
            console.log('appcontroller: _checkUrlParameters');

            this.options.urlParams = $.getUrlVars();

            if (this.options.urlParams !== undefined && this.options.urlParams.length > 0) {
                var p = this.options.urlParams,
                    features = this.options.config.features,
                    tabs = this.options.config.ui.tabs;

                if (p['panels'] !== undefined && p['panels'] == 'no') {
                    // Turn off all UI features.
                    features.banner = false;
                    features.navigation = false;
                    features.splitter = false;
                    features.breadcrumbs = false;

                    // Turn off all navigation tabs too.
                    tabs.tocEnabled = false;
                    tabs.indexEnabled = false;
                    tabs.favoritesEnabled = false;
                    tabs.searchEnabled = false;
                }
            }
        },

        _createView: function () {
            console.log('appcontroller: _createView');

            var self = this;

            $('#' + this.options.targetId).appview({
                title: this.options.config.productFullName,

                features: this.options.config.features,

                ui: this.options.config.ui,

                labels: this.options.config.labels,

                create: function (event) {
                    self._view = $(this).appview('instance');
                },

                buttonclick: function (event, id) {
                    id = id.substring('main-'.length);
                    var handlerName = '_' + id + 'ButtonclickHandler';
                    self[handlerName](event);
                },

                tabbuttonclick: function (event, id) {
                    self._view.showTab(id.substring('tab-button-'.length));
                }
            });
        },

        _createControllers: function () {
            console.log('appcontroller: _createControllers');

            var config = this.options.config,
                controllers = this.options.viewControllers,
                self = this;

            self._createTocController(self, config, controllers);
            self._createFavoritesDataController(config, controllers);
            self._createTopicController(self, config, controllers);
            self._createIndexController(self, config, controllers);
            self._createFavoritesController(self, config, controllers);
            self._createSearchController(self, config, controllers);
        },

        _createTocController: function(self, config, controllers) {
            // Create TOC view controller.
            if (config.ui.tabs.tocEnabled) {
                controllers.tocController = $.adsk.toccontroller({
                    config: config,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesDataController: function(config, controllers) {
            // Create favorites controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesController = $.adsk.favoritescontroller();
            }
        },

        _createTopicController: function(self, config, controllers) {
            // Create topic view controller.
            $.adsk.topiccontroller({
                searchOptions: config.searchOptions,

                favoritesController: controllers.favoritesController,

                create: function() {
                    controllers.topicController = $(this).topiccontroller("instance");
                },

                // Called when topic view are ready to render topic.
                ready: function() {
                    self._defTopicReady.resolve();
                },

                initgotopreviousbutton: function (event, data) {
                    self._view.initGoToPreviousButton(data.enabled);
                },

                initgotoparentbutton: function (event, data) {
                    self._view.initGoToParentButton(data.enabled);
                },

                initgotonextbutton: function (event, data) {
                    self._view.initGoToNextButton(data.enabled);
                },

                initfavoritesbutton: function (event, data) {
                    self._view.initFavoritesButton(data.enabled);
                },

                setupbreadcrumbs: function(event, data) {
                    self._view.updateBreadcrumbs(data.breadcrumbs, self.options.defaultTopicData);
                },

                setuptoc: function(event, data) {
                    if (self.options.config.ui.tabs.tocEnabled) {
                        self.options.viewControllers.tocController.toggleSelectedTocTopic(data);
                    }
                },

                opentopic: function (event, data) {
                    self._handleOpentopic(event, data);
                },

                flagfavorites: function(event, data) {
                    self._view.flagFavorites(data.added);

                    if (controllers.favoritestabController) {
                        controllers.favoritestabController.refresh();
                    }
                }
            });
        },

        _createIndexController: function(self, config, controllers) {
            // Create index view controller.
            if (config.ui.tabs.indexEnabled) {
                controllers.indexController = $.adsk.indexcontroller({
                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesController: function(self, config, controllers) {
            // Create favorites view controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesTabController = $.adsk.favoritestabcontroller({
                    favoritesController: controllers.favoritesController,

                    labels: this.options.config.labels.favorites,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createSearchController: function(self, config, controllers) {
             // Create search view controller.
            if (config.ui.tabs.searchEnabled) {
                $.adsk.searchtabcontroller({
                    config: this.options.config,

                    create: function() {
                        controllers.searchTabController = $(this).searchtabcontroller("instance");
                    },

                    ready: function() {
                        if (self.options.urlParams !== undefined &&
                            self.options.urlParams.length > 0) {

                            var p = self.options.urlParams;

                            if (p['query'] !== undefined &&
                                p['query'] !== '' &&
                                p['panels'] == undefined) {
                                    // Process search query.
                                    controllers.searchTabController.doSearch(decodeURIComponent(p['query']), self);
                                    controllers.searchTabController.closeSearchOptions();
                                    self._view.showTab('search');
                            }

                        }
                    },

                    highlight: function(event, searchOptions) {
                        if (searchOptions && searchOptions.searchQuery) {
                            controllers.topicController.highlightQuery(searchOptions);
                        }
                    },

                    dosearch: function(event) {
                        self._view.showTab('search');
                    }
                });
            }
        },

        _loadAndProcessTOC: function () {
            console.log('appcontroller: _loadAndProcessTOC');

            var controllers = this.options.viewControllers,
                tabs = this.options.config.ui.tabs;

            var tocRootNodesDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.debug('TOC root nodes data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.debug('TOC root nodes data indexed: ', toc);

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocRootNodesData(toc);
                    }
                }, this)
            });

            var tocRootNodesDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeRootData',

                event: 'register-toc-root-node-data',

                loaded: function (event, data) {
                    if (tabs.tocEnabled) {
                        tocRootNodesDataProcessor.setBookTocData(data);
                    }
                }
            });

            // Load TOC root nodes data.
            tocRootNodesDataLoader.loadTocData();

            var tocDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.log('TOC data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.log('TOC data indexed: ', toc);

                    this._setTocData(toc);

                    controllers.topicController.setTocData(toc);

                    if (tabs.searchEnabled) {
                        controllers.searchTabController.setTocData(toc);
                    }

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocData(toc.tocData);
                        controllers.tocController.renderToc();
                    }

                    if (tabs.favoritesEnabled) {
                        controllers.favoritesController.setTocData(toc);
                        controllers.favoritesTabController.refresh();
                    }
                }, this)
            });

            var tocDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeData',

                event: 'register-toc-data',

                loaded: function (event, data) {
                    tocDataProcessor.setBookTocData(data);
                }
            });

            // Load TOC data.
            tocDataLoader.loadTocData();
        },

        _loadContextIdData: function() {
            console.log('appcontroller: _loadContextIdData');

            var self = this;

            // Load context id data.
            var contextIdDataLoader = $.adsk.contextiddataloader({
                config: self.options.config,

                loaded: function (event, data) {
                    $.extend(self.options.config.contextIds, data.bookData);

                    if (data._loadedData == data._totalData) {
                        self._defContextIdsReady.resolve();
                    }
                }
            });

            contextIdDataLoader.loadContextIdData();
        },

        _setTocData: function (data) {
            console.log('appcontroller: _setTocData');

            this.options.toc = data;

            // Set default topic data.
            this.options.defaultTopicData = this._getDefaultTopicData();

            // Show breadcrumbs.
            this._view.updateBreadcrumbs([], this.options.defaultTopicData);

            this._defTocReady.resolve();
        },


        _getDefaultTopicData: function () {
            console.log('appcontroller: _getDefaultTopicData, ');

            var rootTopicData,
               config = this.options.config;

            if (config.defaultPage !== '') {
                rootTopicData = {
                    // TODO: default topic title through config.
                    data: "Home",
                    attr: {
                        href: config.defaultPage,
                        // TODO: description through config.
                        desc: ""
                    }
                }
            } else if (this.options.toc && this.options.toc.tocData) {
                // Try to use first topic data from root and component books TOC data as a default.
                $.each(this.options.toc.tocData, function(bookName, bookTocData) {
                    if (bookTocData[0]) {
                        rootTopicData = bookTocData[0];
                        // break
                        return false;
                    }
                });
            }

            return rootTopicData;
        },


        _handleOpentopic: function (event, data) {
            console.log('appcontroller: _handleOpentopic');

            // Change location hash to open provided topic.
            if (data === undefined || data.href === undefined || data.href == '') {
                return;
            }

            this._setHash('#!/url=' + data.href);
        },

        _addHashchangeHandler: function () {
            console.log('appcontroller: _addHashchangeHandler');

            var self = this;

            // Listen to hash change.
            if ('onhashchange' in window) {
                $(window).on('hashchange', function (event) {
                    self._handleHashchange(event);
                });
            } else {
                setInterval($.proxy(self._handleHashchange, this), 400);
            }
        },

        _handleHashchange: function (event, data) {
            console.log('appcontroller: _handleHashchange');

            var newHash = location.hash;

            this._processHash(newHash);
        },

        _processHash: function (hash) {
            console.log('appcontroller: _processHash, ', hash);

            if (!hash || hash == '') {
                // Load default topic.
                this._handleOpentopic({}, {
                    href: this.options.config.defaultPage
                });
            }

            // If hash starts with hashbang, crop it off.
            var url = this._fetchUrlFromHash(hash);

            if (!url || url == '') {
                // Do nothing.
                return;
            }

            this._processUrl(url);
        },

        _fetchUrlFromHash: function (hash) {
            console.log('appcontroller: _fetchUrlFromHash');

            var url = '';

            if (hash.indexOf('#!') == 0) {
                url = hash.substring(2);
            } else if (hash[0] == '#') {
                url = hash.substring(1);
            } else {
                url = hash;
            }

            return url;
        },

        _processUrl: function (url) {
            console.log('appcontroller: _processUrl: ', url);

            if (url.indexOf('/url=') == 0) {
                // Resolve internal path.
                this._resolveUrl(url);
            }
        },

        _resolveUrl: function (url) {
            console.log('appcontroller: _resolveUrl: ', url);

            var href = url.substring('/url='.length);

            if (!href || href == '') {
                return;
            }

            // Load topic content.
            var data = {
                href: href
            };

            this.options.viewControllers.topicController.openTopic(data);
        },

        _setHash: function (hash) {
            console.log('appcontroller: _setHash: ', hash);

            if (!hash || hash == '') {
                return;
            }

            if (hash[0] != '#') {
                return;
            }

            location.hash = hash;
        },

        _homeButtonclickHandler: function (event) {
            console.log('appcontroller: _homeButtonclickHandler');

            // Open default topic.
            this._handleOpentopic({}, {
                href: this.options.config.defaultPage
            });
        },

        _previousButtonclickHandler: function (event) {
            console.log('appcontroller: _previousButtonclickHandler');

            this.options.viewControllers.topicController.openPreviousTopic();
        },

        _parentButtonclickHandler: function (event) {
            console.log('appcontroller: _parentButtonclickHandler');

            this.options.viewControllers.topicController.openParentTopic();
        },

        _followingButtonclickHandler: function (event) {
            console.log('appcontroller: _followingButtonclickHandler');

            this.options.viewControllers.topicController.openFollowingTopic();
        },

        _favoritesButtonclickHandler: function (event) {
            console.log('appcontroller: _favoritesButtonclickHandler');

            var controllers = this.options.viewControllers;

            controllers.topicController.addOrRemoveTopicFromFavorites();
            controllers.favoritesTabController.refresh();
        },

        _shareButtonclickHandler: function (event) {
            console.log('appcontroller: _shareButtonclickHandler');

            var controller = this.options.viewControllers.shareController;

            if (!controller) {
                controller = this.options.viewControllers.shareController = $.adsk.sharecontroller({
                    //opentopic: $.proxy(this._handleOpentopic, this)
                });
            }

            controller.refresh();
            controller.showShareMenu();
        },

        _printButtonclickHandler: function (event) {
            console.log('appcontroller: _printButtonclickHandler');

            window.print();
        }

        /* Public members. */

    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIbQgYJKoZIhvcNAQcCoIIbMzCCGy8CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // CeKZWGQstNLFioAG7EkYGAv+MHJ7iMdtv/dsrf08MRig
// SIG // ggoWMIIExzCCA6+gAwIBAgIQANMDPwr2vEKYUg3AWz9B
// SIG // yzANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNzA2MjAwMDAw
// SIG // MDBaFw0xODA2MjAyMzU5NTlaMHgxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIDApDYWxpZm9ybmlhMRMwEQYDVQQHDApT
// SIG // YW4gUmFmYWVsMRcwFQYDVQQKDA5BdXRvZGVzaywgSW5j
// SIG // LjENMAsGA1UECwwESVNSQzEXMBUGA1UEAwwOQXV0b2Rl
// SIG // c2ssIEluYy4wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQDEm0ybXlDjakGjfPJQOz8JJyQPZ4frwCIk
// SIG // YuGOWfv+3NZI9Snuy2rQ00G3NuUtmIl0Y0MeypzK5A8s
// SIG // nDGR+bkcRvGn2OmMHS50aggKjsao+rCNMiC8btSwxojg
// SIG // ekzX093J9+/kauZtUESeku1KV2xsBhQ+KTc5TliNgJDJ
// SIG // kB8VW+i2bK0oVx4aMPjX8pivLnbZDYo48U5lQNJtKVBC
// SIG // tnfv0wz2QfRAvXARUxGAnn7xsdgQzYzV6wn+i6N/fMDl
// SIG // e+qWfumXUFSnueYpP85Yi3oNu5kFydgeu34/TTkF1+gD
// SIG // OXfEKLP3cInMUgL7EGTIzHlXaumXFVZobx75J77kq721
// SIG // AgMBAAGjggE+MIIBOjAJBgNVHRMEAjAAMA4GA1UdDwEB
// SIG // /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzBhBgNV
// SIG // HSAEWjBYMFYGBmeBDAEEATBMMCMGCCsGAQUFBwIBFhdo
// SIG // dHRwczovL2Quc3ltY2IuY29tL2NwczAlBggrBgEFBQcC
// SIG // AjAZDBdodHRwczovL2Quc3ltY2IuY29tL3JwYTAfBgNV
// SIG // HSMEGDAWgBTUwAYiSes5S92T4lyhuEd2CXIDWDArBgNV
// SIG // HR8EJDAiMCCgHqAchhpodHRwOi8vcmIuc3ltY2IuY29t
// SIG // L3JiLmNybDBXBggrBgEFBQcBAQRLMEkwHwYIKwYBBQUH
// SIG // MAGGE2h0dHA6Ly9yYi5zeW1jZC5jb20wJgYIKwYBBQUH
// SIG // MAKGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3J0MA0G
// SIG // CSqGSIb3DQEBCwUAA4IBAQApMpNskUmvqk2ygWaCDz+3
// SIG // PTZOZl8Q1FKIoGnv2+iq6wblFUCZt+KgKKyoZoHX6GMd
// SIG // Z/YKeTi5PO4QpKQBgm82kOtKVOdUnRXhUderqaAGrX8W
// SIG // qmXjggda1kRud5e+mdcql+RV1A+RVjjVmSATbeXCC0H5
// SIG // mfvO4o5lnBOfkOQeqKxRjWdEhklOszd+1pQVFX761VyA
// SIG // GfXgr0Jtm9dptQBU1TPybl3+ypIV00IldfMGj1VOKZtd
// SIG // ZrgMiXrdaIWOqRhSEyDdQWvqGzTZm7x5elhF3vcGkRZb
// SIG // 1XqtS7OgDJ7wW0z28NMLA9yfFsPFfOQfMBEEgoCc01P9
// SIG // MBPyiz1XO28kMIIFRzCCBC+gAwIBAgIQfBs1NUrn23Tn
// SIG // QV8RacprqDANBgkqhkiG9w0BAQsFADCBvTELMAkGA1UE
// SIG // BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8w
// SIG // HQYDVQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTow
// SIG // OAYDVQQLEzEoYykgMjAwOCBWZXJpU2lnbiwgSW5jLiAt
// SIG // IEZvciBhdXRob3JpemVkIHVzZSBvbmx5MTgwNgYDVQQD
// SIG // Ey9WZXJpU2lnbiBVbml2ZXJzYWwgUm9vdCBDZXJ0aWZp
// SIG // Y2F0aW9uIEF1dGhvcml0eTAeFw0xNDA3MjIwMDAwMDBa
// SIG // Fw0yNDA3MjEyMzU5NTlaMIGEMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxNTAz
// SIG // BgNVBAMTLFN5bWFudGVjIENsYXNzIDMgU0hBMjU2IENv
// SIG // ZGUgU2lnbmluZyBDQSAtIEcyMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEA15VD1NzfZ645+1KktiYx
// SIG // BHDpt45bKro3aTWVj7vAMOeG2HO73+vRdj+KVo7rLUvw
// SIG // VxhOsY2lM9MLdSPVankn3aPT9w6HZbXerRzx9TW0IlGv
// SIG // IqHBXUuQf8BZTqudeakC1x5JsTtNh/7CeKu/71KunK8I
// SIG // 2TnlmlE+aV8wEE5xY2xY4fAgMxsPdL5byxLh24zEgJRy
// SIG // u/ZFmp7BJQv7oxye2KYJcHHswEdMj33D3hnOPu4Eco4X
// SIG // 0//wsgUyGUzTsByf/qV4IEJwQbAmjG8AyDoAEUF6QbCn
// SIG // ipEEoJl49He082Aq5mxQBLcUYP8NUfSoi4T+IdpcXn31
// SIG // KXlPsER0b21y/wIDAQABo4IBeDCCAXQwLgYIKwYBBQUH
// SIG // AQEEIjAgMB4GCCsGAQUFBzABhhJodHRwOi8vcy5zeW1j
// SIG // ZC5jb20wEgYDVR0TAQH/BAgwBgEB/wIBADBmBgNVHSAE
// SIG // XzBdMFsGC2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEW
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUF
// SIG // BwICMBkaF2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMDYG
// SIG // A1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9zLnN5bWNiLmNv
// SIG // bS91bml2ZXJzYWwtcm9vdC5jcmwwEwYDVR0lBAwwCgYI
// SIG // KwYBBQUHAwMwDgYDVR0PAQH/BAQDAgEGMCkGA1UdEQQi
// SIG // MCCkHjAcMRowGAYDVQQDExFTeW1hbnRlY1BLSS0xLTcy
// SIG // NDAdBgNVHQ4EFgQU1MAGIknrOUvdk+JcobhHdglyA1gw
// SIG // HwYDVR0jBBgwFoAUtnf6aUhHn1MS1cLqBzJ2B9GXBxkw
// SIG // DQYJKoZIhvcNAQELBQADggEBAH/ryqfqi3ZC6z6OIFQw
// SIG // 47e53PpIPhbHD0WVEM0nhqNm8wLtcfiqwlWXkXCD+VJ+
// SIG // Umk8yfHglEaAGLuh1KRWpvMdAJHVhvNIh+DLxDRoIF60
// SIG // y/kF7ZyvcFMnueg+flGgaXGL3FHtgDolMp9Er25DKNMh
// SIG // dbuX2IuLjP6pBEYEhfcVnEsRjcQsF/7Vbn+a4laS8Zaz
// SIG // rS359N/aiZnOsjhEwPdHe8olufoqaDObUHLeqJ/UzSwL
// SIG // NL2LMHhA4I2OJxuQbxq+CBWBXesv4lHnUR7JeCnnHmW/
// SIG // OO8BSgEJJA4WxBR5wUE3NNA9kVKUneFo7wjw4mmcZ26Q
// SIG // CxqTcdQmAsPAWiMxghCEMIIQgAIBATCBmTCBhDELMAkG
// SIG // A1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBv
// SIG // cmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBO
// SIG // ZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAz
// SIG // IFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMgIQANMD
// SIG // Pwr2vEKYUg3AWz9ByzANBglghkgBZQMEAgEFAKB8MBAG
// SIG // CisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgor
// SIG // BgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEE
// SIG // AYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBImAycErTCU+xv
// SIG // bj3RUGD/KhZwTWHNClnrqtG1ztVWpjANBgkqhkiG9w0B
// SIG // AQEFAASCAQCp273SUOMlioQ4a8kXYTajLVXnpCEowmdc
// SIG // xtdMmeM2s04ZGvnAaKVlv9FL7VqSQSf+5U/wHvjrzcET
// SIG // KnMr7TkOn4VN92V3VqRBlmdRGOv1+8slZKEybpkpjAOG
// SIG // +KMj7yxJV5pbXUaunj9s9/8um40q4c8cn3BmmdqufR8t
// SIG // hqG22FYCjMa6BJXTiH/IMPruyTFtriYP/N40KEY9J3hO
// SIG // E/TG1rVVEFDRGnq9/eNl4w0Xt0YZ+DDv4iN3Ya6i0JGX
// SIG // l2AQXN+y3aY+QeJbwWSmhRzVjSKPfJMZ+pMcIReOubT2
// SIG // YsWe6leXNYcGKsCZvKtewB0sGFrK0snvhsHdOZOwaQGa
// SIG // oYIOPTCCDjkGCisGAQQBgjcDAwExgg4pMIIOJQYJKoZI
// SIG // hvcNAQcCoIIOFjCCDhICAQMxDTALBglghkgBZQMEAgEw
// SIG // ggEPBgsqhkiG9w0BCRABBKCB/wSB/DCB+QIBAQYLYIZI
// SIG // AYb4RQEHFwMwMTANBglghkgBZQMEAgEFAAQgODVusDnx
// SIG // Rsil/J3L9nnDl7S1nIvaKtq5/CNBHSYrAFECFQD9M3OL
// SIG // qrzcAGGBkvG0OF+T7jQ9pBgPMjAxODAxMTYwMzA0NTFa
// SIG // MAMCAR6ggYakgYMwgYAxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UE
// SIG // CxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazExMC8GA1UE
// SIG // AxMoU3ltYW50ZWMgU0hBMjU2IFRpbWVTdGFtcGluZyBT
// SIG // aWduZXIgLSBHMqCCCoswggU4MIIEIKADAgECAhB7BbHU
// SIG // SWhRRPfJidKcGZ0SMA0GCSqGSIb3DQEBCwUAMIG9MQsw
// SIG // CQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIElu
// SIG // Yy4xHzAdBgNVBAsTFlZlcmlTaWduIFRydXN0IE5ldHdv
// SIG // cmsxOjA4BgNVBAsTMShjKSAyMDA4IFZlcmlTaWduLCBJ
// SIG // bmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNlIG9ubHkxODA2
// SIG // BgNVBAMTL1ZlcmlTaWduIFVuaXZlcnNhbCBSb290IENl
// SIG // cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE2MDExMjAw
// SIG // MDAwMFoXDTMxMDExMTIzNTk1OVowdzELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MSgwJgYDVQQDEx9TeW1hbnRlYyBTSEEyNTYgVGltZVN0
// SIG // YW1waW5nIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8A
// SIG // MIIBCgKCAQEAu1mdWVVPnYxyXRqBoutV87ABrTxxrDKP
// SIG // BWuGmicAMpdqTclkFEspu8LZKbku7GOz4c8/C1aQ+GIb
// SIG // fuumB+Lef15tQDjUkQbnQXx5HMvLrRu/2JWR8/DubPit
// SIG // ljkuf8EnuHg5xYSl7e2vh47Ojcdt6tKYtTofHjmdw/Sa
// SIG // qPSE4cTRfHHGBim0P+SDDSbDewg+TfkKtzNJ/8o71PWy
// SIG // m0vhiJka9cDpMxTW38eA25Hu/rySV3J39M2ozP4J9ZM3
// SIG // vpWIasXc9LFL1M7oCZFftYR5NYp4rBkyjyPBMkEbWQ6p
// SIG // PrHM+dYr77fY5NUdbRE6kvaTyZzjSO67Uw7UNpeGeMWh
// SIG // NwIDAQABo4IBdzCCAXMwDgYDVR0PAQH/BAQDAgEGMBIG
// SIG // A1UdEwEB/wQIMAYBAf8CAQAwZgYDVR0gBF8wXTBbBgtg
// SIG // hkgBhvhFAQcXAzBMMCMGCCsGAQUFBwIBFhdodHRwczov
// SIG // L2Quc3ltY2IuY29tL2NwczAlBggrBgEFBQcCAjAZGhdo
// SIG // dHRwczovL2Quc3ltY2IuY29tL3JwYTAuBggrBgEFBQcB
// SIG // AQQiMCAwHgYIKwYBBQUHMAGGEmh0dHA6Ly9zLnN5bWNk
// SIG // LmNvbTA2BgNVHR8ELzAtMCugKaAnhiVodHRwOi8vcy5z
// SIG // eW1jYi5jb20vdW5pdmVyc2FsLXJvb3QuY3JsMBMGA1Ud
// SIG // JQQMMAoGCCsGAQUFBwMIMCgGA1UdEQQhMB+kHTAbMRkw
// SIG // FwYDVQQDExBUaW1lU3RhbXAtMjA0OC0zMB0GA1UdDgQW
// SIG // BBSvY9bKo06FcuCnvEHzKaI4f4B1YjAfBgNVHSMEGDAW
// SIG // gBS2d/ppSEefUxLVwuoHMnYH0ZcHGTANBgkqhkiG9w0B
// SIG // AQsFAAOCAQEAdeqwLdU0GVwyRf4O4dRPpnjBb9fq3dxP
// SIG // 86HIgYj3p48V5kApreZd9KLZVmSEcTAq3R5hF2YgVgaY
// SIG // GY1dcfL4l7wJ/RyRR8ni6I0D+8yQL9YKbE4z7Na0k8hM
// SIG // kGNIOUAhxN3WbomYPLWYl+ipBrcJyY9TV0GQL+EeTU7c
// SIG // yhB4bEJu8LbF+GFcUvVO9muN90p6vvPN/QPX2fYDqA/j
// SIG // U/cKdezGdS6qZoUEmbf4Blfhxg726K/a7JsYH6q54zoA
// SIG // v86KlMsB257HOLsPUqvR45QDYApNoP4nbRQy/D+XQOG/
// SIG // mYnb5DkUvdrk08PqK1qzlVhVBH3HmuwjA42FKtL/rqlh
// SIG // gTCCBUswggQzoAMCAQICEFRY8qrXQdZEvISpe6CWUuYw
// SIG // DQYJKoZIhvcNAQELBQAwdzELMAkGA1UEBhMCVVMxHTAb
// SIG // BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYD
// SIG // VQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMSgwJgYD
// SIG // VQQDEx9TeW1hbnRlYyBTSEEyNTYgVGltZVN0YW1waW5n
// SIG // IENBMB4XDTE3MDEwMjAwMDAwMFoXDTI4MDQwMTIzNTk1
// SIG // OVowgYAxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1h
// SIG // bnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50
// SIG // ZWMgVHJ1c3QgTmV0d29yazExMC8GA1UEAxMoU3ltYW50
// SIG // ZWMgU0hBMjU2IFRpbWVTdGFtcGluZyBTaWduZXIgLSBH
// SIG // MjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB
// SIG // AJnz/NgECQOG+ddcppPAQnzqfGPPXQDijvPAkN+PKfUY
// SIG // 6pS3kuXXsKBzgejpCptKfAH/nY+kOacO6kX0Igw6cO05
// SIG // RYvkxRtc8EVoRiQFY3abHPyebCqxVuWKf1JxrvI11UYj
// SIG // BhzPSC0dtM242XYjjhz/Pr+7BlxpB6ZlDvhern0u7U2u
// SIG // Ne/J1wBC/SiVDp9dckIJvMPaRNLtzEeE5PzKLaxYvq73
// SIG // rtlEDQi3wnfWGkNw0W4D3lKSxBAIcdm6IlXyH7ztm507
// SIG // 4l4dTIP/lw97C+dVg07SDeu+1+yubke5n9+l1lG8BFXt
// SIG // /ydwTMntKksT4bG5TA/JAe5VZV9pAnhmyz8CAwEAAaOC
// SIG // AccwggHDMAwGA1UdEwEB/wQCMAAwZgYDVR0gBF8wXTBb
// SIG // BgtghkgBhvhFAQcXAzBMMCMGCCsGAQUFBwIBFhdodHRw
// SIG // czovL2Quc3ltY2IuY29tL2NwczAlBggrBgEFBQcCAjAZ
// SIG // GhdodHRwczovL2Quc3ltY2IuY29tL3JwYTBABgNVHR8E
// SIG // OTA3MDWgM6Axhi9odHRwOi8vdHMtY3JsLndzLnN5bWFu
// SIG // dGVjLmNvbS9zaGEyNTYtdHNzLWNhLmNybDAWBgNVHSUB
// SIG // Af8EDDAKBggrBgEFBQcDCDAOBgNVHQ8BAf8EBAMCB4Aw
// SIG // dwYIKwYBBQUHAQEEazBpMCoGCCsGAQUFBzABhh5odHRw
// SIG // Oi8vdHMtb2NzcC53cy5zeW1hbnRlYy5jb20wOwYIKwYB
// SIG // BQUHMAKGL2h0dHA6Ly90cy1haWEud3Muc3ltYW50ZWMu
// SIG // Y29tL3NoYTI1Ni10c3MtY2EuY2VyMCgGA1UdEQQhMB+k
// SIG // HTAbMRkwFwYDVQQDExBUaW1lU3RhbXAtMjA0OC01MB0G
// SIG // A1UdDgQWBBQJtcH+lnKXKUOayeACuq74/S+69jAfBgNV
// SIG // HSMEGDAWgBSvY9bKo06FcuCnvEHzKaI4f4B1YjANBgkq
// SIG // hkiG9w0BAQsFAAOCAQEAF7MKiOlcWl4gazsKFbJsxamK
// SIG // MofTsfQcU66Fvj+b/9e8t5SFtMdSfpTove1hstSnmeTD
// SIG // yZPBNT0L6GgKXVaYvbEiO9FEete/8G1RMorVI984ATf2
// SIG // 4lMreisRj7dNbHozAxt8awmUF7vk21jUIRNl5+zRJcos
// SIG // dZqcf/zJuypoq8R9tM+jyWyn2cQAnIkKd5H0TaL7MTuG
// SIG // bvbmH1ADhpu/y0Kr5nabcloRAYrG76VvlefdrrrmImXw
// SIG // GFkbEcnNgLfYl0cfQgj4rHEfsEZTs9Sy1aOrUHVIEheC
// SIG // rc/gQU8yfs2VHL+Rigg9pKdnApbfJEyl0EHAgmCjihcy
// SIG // S9O8z6S0jDGCAlowggJWAgEBMIGLMHcxCzAJBgNVBAYT
// SIG // AlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlv
// SIG // bjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29y
// SIG // azEoMCYGA1UEAxMfU3ltYW50ZWMgU0hBMjU2IFRpbWVT
// SIG // dGFtcGluZyBDQQIQVFjyqtdB1kS8hKl7oJZS5jALBglg
// SIG // hkgBZQMEAgGggaQwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3
// SIG // DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0xODAxMTYwMzA0
// SIG // NTFaMC8GCSqGSIb3DQEJBDEiBCAff7ErQrsxbw9TZj7u
// SIG // EKsbFe4pSUarbxz466OoFTTYuDA3BgsqhkiG9w0BCRAC
// SIG // LzEoMCYwJDAiBCDPesF60Efs1f3DaCIDGxLU7weLbytM
// SIG // XmukH4/yz0utZzALBgkqhkiG9w0BAQEEggEAA2F3z+Wd
// SIG // G59lRPNf4UrHGOYOmr7iKlb1h/4KEJ5Iz5BtROmeRk7d
// SIG // BAG/2OsKLUHca5aKoq/vmA8zYmLKiADCGY/A/OHI/+/h
// SIG // OyY+8uU7WEDh6fopAw+xUC/8tIG5/C/1fY8dbRcaEHEb
// SIG // xilDK7oRXm+mcrRiVA4OXw8sH/o9ps69G6aSnjZc2OGG
// SIG // 6G8zHz0f0YeXE7HgBmZNu5U3imUvlqdHkg+QsmFKy9Br
// SIG // KdwyI5iptDLwvlxfc93wfdIp4YaJGTKLjF7UDV4nHm4w
// SIG // w2hG3cEEInATPtLPyJnBhAlnEzJq0ByeAvGLB5btGkj0
// SIG // /aWnOZWIm2Kx5vHYXHpNwDOKgQ==
// SIG // End signature block
