// App controller.
(function ($) {
    $.widget('adsk.appcontroller', {
        options: {
            config: {
                productFullName: '',

                ui: {
                    tabs: {
                        order: [
                            'C','I','S','F'
                        ],
                        tocEnabled: true,
                        indexEnabled: true,
                        favoritesEnabled: true,
                        searchEnabled: true
                    },

                    tools: {
                        order: ['home', 'previous', 'parent', 'following', 'separator',
                                'favorites', 'share', 'separator', 'print'],
                        home: true,
                        previous: true,
                        parent: true,
                        following: true,
                        favorites: true,
                        share: true,
                        print: true
                    },

                    responsive: true
                },

                features: {
                    tabsbarParent: 'banner',
                    banner: true,
                    header: true,
                    toolbar: true,
                    breadcrumbs: true,
                    tabsbar: true,
                    navigation: true,
                    splitter: true
                }
            },
            targetId: 'browser',
            viewControllers: {},
            toc: {},
            defaultTopicData: undefined,
            urlParams: undefined
        },

        /* Private members. */

        _view: undefined,
        _defTopicReady: new $.Deferred(),
        _defTocReady: new $.Deferred(),
        _defContextIdsReady: new $.Deferred(),

        _create: function () {
            console.log('appcontroller: _create');

            var self = this;

            this._setupPolyfills();

            $.when(this._defTopicReady, this._defTocReady, this._defContextIdsReady).done(function() {
                self._addHashchangeHandler();

                // Go to default topic page if there was no hash data specified by the user.
                if (location.hash == '') {
                    var href = self.options.defaultTopicData.attr.href;

                    if (self.options.urlParams !== undefined &&
                        self.options.urlParams.length > 0) {

                        var p = self.options.urlParams;

                        // Process contextId query.
                        if (p['contextId'] !== undefined &&
                            p['contextId'] !== '' ) {

                                // Open specified topic.
                                var id = decodeURIComponent(p['contextId']).toUpperCase(),
                                    contextIdHref = self.options.config.contextIds[id];

                                if (contextIdHref !== undefined && contextIdHref != '') {
                                    href = './'.concat(contextIdHref);
                                }
                        }
                    }

                    // Go to topic page.
                    self._handleOpentopic({}, { href: href });
                } else {
                    $(window).trigger('hashchange');
                }
            });

            this._checkUrlParameters();

            this._createView();

            this._createControllers();

            // Hide main preloader.
            this._view.hidePreloader();

            // Load context id data.
            this._loadContextIdData();

            // Load and process TOC data.
            this._loadAndProcessTOC();
        },

        _setupPolyfills: function() {
            $('html').addClass($.fn.details.support ? 'details' : 'no-details');
        },

        _checkUrlParameters: function() {
            console.log('appcontroller: _checkUrlParameters');

            this.options.urlParams = $.getUrlVars();

            if (this.options.urlParams !== undefined && this.options.urlParams.length > 0) {
                var p = this.options.urlParams,
                    features = this.options.config.features,
                    tabs = this.options.config.ui.tabs;

                if (p['panels'] !== undefined && p['panels'] == 'no') {
                    // Turn off all UI features.
                    features.banner = false;
                    features.navigation = false;
                    features.splitter = false;
                    features.breadcrumbs = false;

                    // Turn off all navigation tabs too.
                    tabs.tocEnabled = false;
                    tabs.indexEnabled = false;
                    tabs.favoritesEnabled = false;
                    tabs.searchEnabled = false;
                }
            }
        },

        _createView: function () {
            console.log('appcontroller: _createView');

            var self = this;

            $('#' + this.options.targetId).appview({
                title: this.options.config.productFullName,

                features: this.options.config.features,

                ui: this.options.config.ui,

                labels: this.options.config.labels,

                create: function (event) {
                    self._view = $(this).appview('instance');
                },

                buttonclick: function (event, id) {
                    id = id.substring('main-'.length);
                    var handlerName = '_' + id + 'ButtonclickHandler';
                    self[handlerName](event);
                },

                tabbuttonclick: function (event, id) {
                    self._view.showTab(id.substring('tab-button-'.length));
                }
            });
        },

        _createControllers: function () {
            console.log('appcontroller: _createControllers');

            var config = this.options.config,
                controllers = this.options.viewControllers,
                self = this;

            self._createTocController(self, config, controllers);
            self._createFavoritesDataController(config, controllers);
            self._createTopicController(self, config, controllers);
            self._createIndexController(self, config, controllers);
            self._createFavoritesController(self, config, controllers);
            self._createSearchController(self, config, controllers);
        },

        _createTocController: function(self, config, controllers) {
            // Create TOC view controller.
            if (config.ui.tabs.tocEnabled) {
                controllers.tocController = $.adsk.toccontroller({
                    config: config,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesDataController: function(config, controllers) {
            // Create favorites controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesController = $.adsk.favoritescontroller();
            }
        },

        _createTopicController: function(self, config, controllers) {
            // Create topic view controller.
            $.adsk.topiccontroller({
                searchOptions: config.searchOptions,

                favoritesController: controllers.favoritesController,

                create: function() {
                    controllers.topicController = $(this).topiccontroller("instance");
                },

                // Called when topic view are ready to render topic.
                ready: function() {
                    self._defTopicReady.resolve();
                },

                initgotopreviousbutton: function (event, data) {
                    self._view.initGoToPreviousButton(data.enabled);
                },

                initgotoparentbutton: function (event, data) {
                    self._view.initGoToParentButton(data.enabled);
                },

                initgotonextbutton: function (event, data) {
                    self._view.initGoToNextButton(data.enabled);
                },

                initfavoritesbutton: function (event, data) {
                    self._view.initFavoritesButton(data.enabled);
                },

                setupbreadcrumbs: function(event, data) {
                    self._view.updateBreadcrumbs(data.breadcrumbs, self.options.defaultTopicData);
                },

                setuptoc: function(event, data) {
                    if (self.options.config.ui.tabs.tocEnabled) {
                        self.options.viewControllers.tocController.toggleSelectedTocTopic(data);
                    }
                },

                opentopic: function (event, data) {
                    self._handleOpentopic(event, data);
                },

                flagfavorites: function(event, data) {
                    self._view.flagFavorites(data.added);

                    if (controllers.favoritestabController) {
                        controllers.favoritestabController.refresh();
                    }
                }
            });
        },

        _createIndexController: function(self, config, controllers) {
            // Create index view controller.
            if (config.ui.tabs.indexEnabled) {
                controllers.indexController = $.adsk.indexcontroller({
                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesController: function(self, config, controllers) {
            // Create favorites view controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesTabController = $.adsk.favoritestabcontroller({
                    favoritesController: controllers.favoritesController,

                    labels: this.options.config.labels.favorites,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createSearchController: function(self, config, controllers) {
             // Create search view controller.
            if (config.ui.tabs.searchEnabled) {
                $.adsk.searchtabcontroller({
                    config: this.options.config,

                    create: function() {
                        controllers.searchTabController = $(this).searchtabcontroller("instance");
                    },

                    ready: function() {
                        if (self.options.urlParams !== undefined &&
                            self.options.urlParams.length > 0) {

                            var p = self.options.urlParams;

                            if (p['query'] !== undefined &&
                                p['query'] !== '' &&
                                p['panels'] == undefined) {
                                    // Process search query.
                                    controllers.searchTabController.doSearch(decodeURIComponent(p['query']), self);
                                    controllers.searchTabController.closeSearchOptions();
                                    self._view.showTab('search');
                            }

                        }
                    },

                    highlight: function(event, searchOptions) {
                        if (searchOptions && searchOptions.searchQuery) {
                            controllers.topicController.highlightQuery(searchOptions);
                        }
                    },

                    dosearch: function(event) {
                        self._view.showTab('search');
                    }
                });
            }
        },

        _loadAndProcessTOC: function () {
            console.log('appcontroller: _loadAndProcessTOC');

            var controllers = this.options.viewControllers,
                tabs = this.options.config.ui.tabs;

            var tocRootNodesDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.debug('TOC root nodes data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.debug('TOC root nodes data indexed: ', toc);

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocRootNodesData(toc);
                    }
                }, this)
            });

            var tocRootNodesDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeRootData',

                event: 'register-toc-root-node-data',

                loaded: function (event, data) {
                    if (tabs.tocEnabled) {
                        tocRootNodesDataProcessor.setBookTocData(data);
                    }
                }
            });

            // Load TOC root nodes data.
            tocRootNodesDataLoader.loadTocData();

            var tocDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.log('TOC data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.log('TOC data indexed: ', toc);

                    this._setTocData(toc);

                    controllers.topicController.setTocData(toc);

                    if (tabs.searchEnabled) {
                        controllers.searchTabController.setTocData(toc);
                    }

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocData(toc.tocData);
                        controllers.tocController.renderToc();
                    }

                    if (tabs.favoritesEnabled) {
                        controllers.favoritesController.setTocData(toc);
                        controllers.favoritesTabController.refresh();
                    }
                }, this)
            });

            var tocDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeData',

                event: 'register-toc-data',

                loaded: function (event, data) {
                    tocDataProcessor.setBookTocData(data);
                }
            });

            // Load TOC data.
            tocDataLoader.loadTocData();
        },

        _loadContextIdData: function() {
            console.log('appcontroller: _loadContextIdData');

            var self = this;

            // Load context id data.
            var contextIdDataLoader = $.adsk.contextiddataloader({
                config: self.options.config,

                loaded: function (event, data) {
                    $.extend(self.options.config.contextIds, data.bookData);

                    if (data._loadedData == data._totalData) {
                        self._defContextIdsReady.resolve();
                    }
                }
            });

            contextIdDataLoader.loadContextIdData();
        },

        _setTocData: function (data) {
            console.log('appcontroller: _setTocData');

            this.options.toc = data;

            // Set default topic data.
            this.options.defaultTopicData = this._getDefaultTopicData();

            // Show breadcrumbs.
            this._view.updateBreadcrumbs([], this.options.defaultTopicData);

            this._defTocReady.resolve();
        },


        _getDefaultTopicData: function () {
            console.log('appcontroller: _getDefaultTopicData, ');

            var rootTopicData,
               config = this.options.config;

            if (config.defaultPage !== '') {
                rootTopicData = {
                    // TODO: default topic title through config.
                    data: "Home",
                    attr: {
                        href: config.defaultPage,
                        // TODO: description through config.
                        desc: ""
                    }
                }
            } else if (this.options.toc && this.options.toc.tocData) {
                // Try to use first topic data from root and component books TOC data as a default.
                $.each(this.options.toc.tocData, function(bookName, bookTocData) {
                    if (bookTocData[0]) {
                        rootTopicData = bookTocData[0];
                        // break
                        return false;
                    }
                });
            }

            return rootTopicData;
        },


        _handleOpentopic: function (event, data) {
            console.log('appcontroller: _handleOpentopic');

            // Change location hash to open provided topic.
            if (data === undefined || data.href === undefined || data.href == '') {
                return;
            }

            this._setHash('#!/url=' + data.href);
        },

        _addHashchangeHandler: function () {
            console.log('appcontroller: _addHashchangeHandler');

            var self = this;

            // Listen to hash change.
            if ('onhashchange' in window) {
                $(window).on('hashchange', function (event) {
                    self._handleHashchange(event);
                });
            } else {
                setInterval($.proxy(self._handleHashchange, this), 400);
            }
        },

        _handleHashchange: function (event, data) {
            console.log('appcontroller: _handleHashchange');

            var newHash = location.hash;

            this._processHash(newHash);
        },

        _processHash: function (hash) {
            console.log('appcontroller: _processHash, ', hash);

            if (!hash || hash == '') {
                // Load default topic.
                this._handleOpentopic({}, {
                    href: this.options.config.defaultPage
                });
            }

            // If hash starts with hashbang, crop it off.
            var url = this._fetchUrlFromHash(hash);

            if (!url || url == '') {
                // Do nothing.
                return;
            }

            this._processUrl(url);
        },

        _fetchUrlFromHash: function (hash) {
            console.log('appcontroller: _fetchUrlFromHash');

            var url = '';

            if (hash.indexOf('#!') == 0) {
                url = hash.substring(2);
            } else if (hash[0] == '#') {
                url = hash.substring(1);
            } else {
                url = hash;
            }

            return url;
        },

        _processUrl: function (url) {
            console.log('appcontroller: _processUrl: ', url);

            if (url.indexOf('/url=') == 0) {
                // Resolve internal path.
                this._resolveUrl(url);
            }
        },

        _resolveUrl: function (url) {
            console.log('appcontroller: _resolveUrl: ', url);

            var href = url.substring('/url='.length);

            if (!href || href == '') {
                return;
            }

            // Load topic content.
            var data = {
                href: href
            };

            this.options.viewControllers.topicController.openTopic(data);
        },

        _setHash: function (hash) {
            console.log('appcontroller: _setHash: ', hash);

            if (!hash || hash == '') {
                return;
            }

            if (hash[0] != '#') {
                return;
            }

            location.hash = hash;
        },

        _homeButtonclickHandler: function (event) {
            console.log('appcontroller: _homeButtonclickHandler');

            // Open default topic.
            this._handleOpentopic({}, {
                href: this.options.config.defaultPage
            });
        },

        _previousButtonclickHandler: function (event) {
            console.log('appcontroller: _previousButtonclickHandler');

            this.options.viewControllers.topicController.openPreviousTopic();
        },

        _parentButtonclickHandler: function (event) {
            console.log('appcontroller: _parentButtonclickHandler');

            this.options.viewControllers.topicController.openParentTopic();
        },

        _followingButtonclickHandler: function (event) {
            console.log('appcontroller: _followingButtonclickHandler');

            this.options.viewControllers.topicController.openFollowingTopic();
        },

        _favoritesButtonclickHandler: function (event) {
            console.log('appcontroller: _favoritesButtonclickHandler');

            var controllers = this.options.viewControllers;

            controllers.topicController.addOrRemoveTopicFromFavorites();
            controllers.favoritesTabController.refresh();
        },

        _shareButtonclickHandler: function (event) {
            console.log('appcontroller: _shareButtonclickHandler');

            var controller = this.options.viewControllers.shareController;

            if (!controller) {
                controller = this.options.viewControllers.shareController = $.adsk.sharecontroller({
                    //opentopic: $.proxy(this._handleOpentopic, this)
                });
            }

            controller.refresh();
            controller.showShareMenu();
        },

        _printButtonclickHandler: function (event) {
            console.log('appcontroller: _printButtonclickHandler');

            window.print();
        }

        /* Public members. */

    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIbQQYJKoZIhvcNAQcCoIIbMjCCGy4CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // CeKZWGQstNLFioAG7EkYGAv+MHJ7iMdtv/dsrf08MRig
// SIG // ggoWMIIExzCCA6+gAwIBAgIQANMDPwr2vEKYUg3AWz9B
// SIG // yzANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNzA2MjAwMDAw
// SIG // MDBaFw0xODA2MjAyMzU5NTlaMHgxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIDApDYWxpZm9ybmlhMRMwEQYDVQQHDApT
// SIG // YW4gUmFmYWVsMRcwFQYDVQQKDA5BdXRvZGVzaywgSW5j
// SIG // LjENMAsGA1UECwwESVNSQzEXMBUGA1UEAwwOQXV0b2Rl
// SIG // c2ssIEluYy4wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQDEm0ybXlDjakGjfPJQOz8JJyQPZ4frwCIk
// SIG // YuGOWfv+3NZI9Snuy2rQ00G3NuUtmIl0Y0MeypzK5A8s
// SIG // nDGR+bkcRvGn2OmMHS50aggKjsao+rCNMiC8btSwxojg
// SIG // ekzX093J9+/kauZtUESeku1KV2xsBhQ+KTc5TliNgJDJ
// SIG // kB8VW+i2bK0oVx4aMPjX8pivLnbZDYo48U5lQNJtKVBC
// SIG // tnfv0wz2QfRAvXARUxGAnn7xsdgQzYzV6wn+i6N/fMDl
// SIG // e+qWfumXUFSnueYpP85Yi3oNu5kFydgeu34/TTkF1+gD
// SIG // OXfEKLP3cInMUgL7EGTIzHlXaumXFVZobx75J77kq721
// SIG // AgMBAAGjggE+MIIBOjAJBgNVHRMEAjAAMA4GA1UdDwEB
// SIG // /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzBhBgNV
// SIG // HSAEWjBYMFYGBmeBDAEEATBMMCMGCCsGAQUFBwIBFhdo
// SIG // dHRwczovL2Quc3ltY2IuY29tL2NwczAlBggrBgEFBQcC
// SIG // AjAZDBdodHRwczovL2Quc3ltY2IuY29tL3JwYTAfBgNV
// SIG // HSMEGDAWgBTUwAYiSes5S92T4lyhuEd2CXIDWDArBgNV
// SIG // HR8EJDAiMCCgHqAchhpodHRwOi8vcmIuc3ltY2IuY29t
// SIG // L3JiLmNybDBXBggrBgEFBQcBAQRLMEkwHwYIKwYBBQUH
// SIG // MAGGE2h0dHA6Ly9yYi5zeW1jZC5jb20wJgYIKwYBBQUH
// SIG // MAKGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3J0MA0G
// SIG // CSqGSIb3DQEBCwUAA4IBAQApMpNskUmvqk2ygWaCDz+3
// SIG // PTZOZl8Q1FKIoGnv2+iq6wblFUCZt+KgKKyoZoHX6GMd
// SIG // Z/YKeTi5PO4QpKQBgm82kOtKVOdUnRXhUderqaAGrX8W
// SIG // qmXjggda1kRud5e+mdcql+RV1A+RVjjVmSATbeXCC0H5
// SIG // mfvO4o5lnBOfkOQeqKxRjWdEhklOszd+1pQVFX761VyA
// SIG // GfXgr0Jtm9dptQBU1TPybl3+ypIV00IldfMGj1VOKZtd
// SIG // ZrgMiXrdaIWOqRhSEyDdQWvqGzTZm7x5elhF3vcGkRZb
// SIG // 1XqtS7OgDJ7wW0z28NMLA9yfFsPFfOQfMBEEgoCc01P9
// SIG // MBPyiz1XO28kMIIFRzCCBC+gAwIBAgIQfBs1NUrn23Tn
// SIG // QV8RacprqDANBgkqhkiG9w0BAQsFADCBvTELMAkGA1UE
// SIG // BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8w
// SIG // HQYDVQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTow
// SIG // OAYDVQQLEzEoYykgMjAwOCBWZXJpU2lnbiwgSW5jLiAt
// SIG // IEZvciBhdXRob3JpemVkIHVzZSBvbmx5MTgwNgYDVQQD
// SIG // Ey9WZXJpU2lnbiBVbml2ZXJzYWwgUm9vdCBDZXJ0aWZp
// SIG // Y2F0aW9uIEF1dGhvcml0eTAeFw0xNDA3MjIwMDAwMDBa
// SIG // Fw0yNDA3MjEyMzU5NTlaMIGEMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxNTAz
// SIG // BgNVBAMTLFN5bWFudGVjIENsYXNzIDMgU0hBMjU2IENv
// SIG // ZGUgU2lnbmluZyBDQSAtIEcyMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEA15VD1NzfZ645+1KktiYx
// SIG // BHDpt45bKro3aTWVj7vAMOeG2HO73+vRdj+KVo7rLUvw
// SIG // VxhOsY2lM9MLdSPVankn3aPT9w6HZbXerRzx9TW0IlGv
// SIG // IqHBXUuQf8BZTqudeakC1x5JsTtNh/7CeKu/71KunK8I
// SIG // 2TnlmlE+aV8wEE5xY2xY4fAgMxsPdL5byxLh24zEgJRy
// SIG // u/ZFmp7BJQv7oxye2KYJcHHswEdMj33D3hnOPu4Eco4X
// SIG // 0//wsgUyGUzTsByf/qV4IEJwQbAmjG8AyDoAEUF6QbCn
// SIG // ipEEoJl49He082Aq5mxQBLcUYP8NUfSoi4T+IdpcXn31
// SIG // KXlPsER0b21y/wIDAQABo4IBeDCCAXQwLgYIKwYBBQUH
// SIG // AQEEIjAgMB4GCCsGAQUFBzABhhJodHRwOi8vcy5zeW1j
// SIG // ZC5jb20wEgYDVR0TAQH/BAgwBgEB/wIBADBmBgNVHSAE
// SIG // XzBdMFsGC2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEW
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUF
// SIG // BwICMBkaF2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMDYG
// SIG // A1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9zLnN5bWNiLmNv
// SIG // bS91bml2ZXJzYWwtcm9vdC5jcmwwEwYDVR0lBAwwCgYI
// SIG // KwYBBQUHAwMwDgYDVR0PAQH/BAQDAgEGMCkGA1UdEQQi
// SIG // MCCkHjAcMRowGAYDVQQDExFTeW1hbnRlY1BLSS0xLTcy
// SIG // NDAdBgNVHQ4EFgQU1MAGIknrOUvdk+JcobhHdglyA1gw
// SIG // HwYDVR0jBBgwFoAUtnf6aUhHn1MS1cLqBzJ2B9GXBxkw
// SIG // DQYJKoZIhvcNAQELBQADggEBAH/ryqfqi3ZC6z6OIFQw
// SIG // 47e53PpIPhbHD0WVEM0nhqNm8wLtcfiqwlWXkXCD+VJ+
// SIG // Umk8yfHglEaAGLuh1KRWpvMdAJHVhvNIh+DLxDRoIF60
// SIG // y/kF7ZyvcFMnueg+flGgaXGL3FHtgDolMp9Er25DKNMh
// SIG // dbuX2IuLjP6pBEYEhfcVnEsRjcQsF/7Vbn+a4laS8Zaz
// SIG // rS359N/aiZnOsjhEwPdHe8olufoqaDObUHLeqJ/UzSwL
// SIG // NL2LMHhA4I2OJxuQbxq+CBWBXesv4lHnUR7JeCnnHmW/
// SIG // OO8BSgEJJA4WxBR5wUE3NNA9kVKUneFo7wjw4mmcZ26Q
// SIG // CxqTcdQmAsPAWiMxghCDMIIQfwIBATCBmTCBhDELMAkG
// SIG // A1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBv
// SIG // cmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBO
// SIG // ZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAz
// SIG // IFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMgIQANMD
// SIG // Pwr2vEKYUg3AWz9ByzANBglghkgBZQMEAgEFAKB8MBAG
// SIG // CisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgor
// SIG // BgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEE
// SIG // AYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBImAycErTCU+xv
// SIG // bj3RUGD/KhZwTWHNClnrqtG1ztVWpjANBgkqhkiG9w0B
// SIG // AQEFAASCAQCp273SUOMlioQ4a8kXYTajLVXnpCEowmdc
// SIG // xtdMmeM2s04ZGvnAaKVlv9FL7VqSQSf+5U/wHvjrzcET
// SIG // KnMr7TkOn4VN92V3VqRBlmdRGOv1+8slZKEybpkpjAOG
// SIG // +KMj7yxJV5pbXUaunj9s9/8um40q4c8cn3BmmdqufR8t
// SIG // hqG22FYCjMa6BJXTiH/IMPruyTFtriYP/N40KEY9J3hO
// SIG // E/TG1rVVEFDRGnq9/eNl4w0Xt0YZ+DDv4iN3Ya6i0JGX
// SIG // l2AQXN+y3aY+QeJbwWSmhRzVjSKPfJMZ+pMcIReOubT2
// SIG // YsWe6leXNYcGKsCZvKtewB0sGFrK0snvhsHdOZOwaQGa
// SIG // oYIOPDCCDjgGCisGAQQBgjcDAwExgg4oMIIOJAYJKoZI
// SIG // hvcNAQcCoIIOFTCCDhECAQMxDTALBglghkgBZQMEAgEw
// SIG // ggEOBgsqhkiG9w0BCRABBKCB/gSB+zCB+AIBAQYLYIZI
// SIG // AYb4RQEHFwMwMTANBglghkgBZQMEAgEFAAQgODVusDnx
// SIG // Rsil/J3L9nnDl7S1nIvaKtq5/CNBHSYrAFECFBMrrCiA
// SIG // 4OKGAbfr8PNZg5iczS77GA8yMDE4MDExNjAzMDMzMFow
// SIG // AwIBHqCBhqSBgzCBgDELMAkGA1UEBhMCVVMxHTAbBgNV
// SIG // BAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQL
// SIG // ExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTEwLwYDVQQD
// SIG // EyhTeW1hbnRlYyBTSEEyNTYgVGltZVN0YW1waW5nIFNp
// SIG // Z25lciAtIEcyoIIKizCCBTgwggQgoAMCAQICEHsFsdRJ
// SIG // aFFE98mJ0pwZnRIwDQYJKoZIhvcNAQELBQAwgb0xCzAJ
// SIG // BgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5j
// SIG // LjEfMB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29y
// SIG // azE6MDgGA1UECxMxKGMpIDIwMDggVmVyaVNpZ24sIElu
// SIG // Yy4gLSBGb3IgYXV0aG9yaXplZCB1c2Ugb25seTE4MDYG
// SIG // A1UEAxMvVmVyaVNpZ24gVW5pdmVyc2FsIFJvb3QgQ2Vy
// SIG // dGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMTYwMTEyMDAw
// SIG // MDAwWhcNMzEwMTExMjM1OTU5WjB3MQswCQYDVQQGEwJV
// SIG // UzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24x
// SIG // HzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsx
// SIG // KDAmBgNVBAMTH1N5bWFudGVjIFNIQTI1NiBUaW1lU3Rh
// SIG // bXBpbmcgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQC7WZ1ZVU+djHJdGoGi61XzsAGtPHGsMo8F
// SIG // a4aaJwAyl2pNyWQUSym7wtkpuS7sY7Phzz8LVpD4Yht+
// SIG // 66YH4t5/Xm1AONSRBudBfHkcy8utG7/YlZHz8O5s+K2W
// SIG // OS5/wSe4eDnFhKXt7a+Hjs6Nx23q0pi1Oh8eOZ3D9Jqo
// SIG // 9IThxNF8ccYGKbQ/5IMNJsN7CD5N+Qq3M0n/yjvU9bKb
// SIG // S+GImRr1wOkzFNbfx4Dbke7+vJJXcnf0zajM/gn1kze+
// SIG // lYhqxdz0sUvUzugJkV+1hHk1inisGTKPI8EyQRtZDqk+
// SIG // scz51ivvt9jk1R1tETqS9pPJnONI7rtTDtQ2l4Z4xaE3
// SIG // AgMBAAGjggF3MIIBczAOBgNVHQ8BAf8EBAMCAQYwEgYD
// SIG // VR0TAQH/BAgwBgEB/wIBADBmBgNVHSAEXzBdMFsGC2CG
// SIG // SAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8v
// SIG // ZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkaF2h0
// SIG // dHBzOi8vZC5zeW1jYi5jb20vcnBhMC4GCCsGAQUFBwEB
// SIG // BCIwIDAeBggrBgEFBQcwAYYSaHR0cDovL3Muc3ltY2Qu
// SIG // Y29tMDYGA1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9zLnN5
// SIG // bWNiLmNvbS91bml2ZXJzYWwtcm9vdC5jcmwwEwYDVR0l
// SIG // BAwwCgYIKwYBBQUHAwgwKAYDVR0RBCEwH6QdMBsxGTAX
// SIG // BgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTMwHQYDVR0OBBYE
// SIG // FK9j1sqjToVy4Ke8QfMpojh/gHViMB8GA1UdIwQYMBaA
// SIG // FLZ3+mlIR59TEtXC6gcydgfRlwcZMA0GCSqGSIb3DQEB
// SIG // CwUAA4IBAQB16rAt1TQZXDJF/g7h1E+meMFv1+rd3E/z
// SIG // ociBiPenjxXmQCmt5l30otlWZIRxMCrdHmEXZiBWBpgZ
// SIG // jV1x8viXvAn9HJFHyeLojQP7zJAv1gpsTjPs1rSTyEyQ
// SIG // Y0g5QCHE3dZuiZg8tZiX6KkGtwnJj1NXQZAv4R5NTtzK
// SIG // EHhsQm7wtsX4YVxS9U72a433Snq+8839A9fZ9gOoD+NT
// SIG // 9wp17MZ1LqpmhQSZt/gGV+HGDvbor9rsmxgfqrnjOgC/
// SIG // zoqUywHbnsc4uw9Sq9HjlANgCk2g/idtFDL8P5dA4b+Z
// SIG // idvkORS92uTTw+orWrOVWFUEfcea7CMDjYUq0v+uqWGB
// SIG // MIIFSzCCBDOgAwIBAgIQVFjyqtdB1kS8hKl7oJZS5jAN
// SIG // BgkqhkiG9w0BAQsFADB3MQswCQYDVQQGEwJVUzEdMBsG
// SIG // A1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNV
// SIG // BAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxKDAmBgNV
// SIG // BAMTH1N5bWFudGVjIFNIQTI1NiBUaW1lU3RhbXBpbmcg
// SIG // Q0EwHhcNMTcwMTAyMDAwMDAwWhcNMjgwNDAxMjM1OTU5
// SIG // WjCBgDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFu
// SIG // dGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRl
// SIG // YyBUcnVzdCBOZXR3b3JrMTEwLwYDVQQDEyhTeW1hbnRl
// SIG // YyBTSEEyNTYgVGltZVN0YW1waW5nIFNpZ25lciAtIEcy
// SIG // MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
// SIG // mfP82AQJA4b511ymk8BCfOp8Y89dAOKO88CQ348p9Rjq
// SIG // lLeS5dewoHOB6OkKm0p8Af+dj6Q5pw7qRfQiDDpw7TlF
// SIG // i+TFG1zwRWhGJAVjdpsc/J5sKrFW5Yp/UnGu8jXVRiMG
// SIG // HM9ILR20zbjZdiOOHP8+v7sGXGkHpmUO+F6ufS7tTa41
// SIG // 78nXAEL9KJUOn11yQgm8w9pE0u3MR4Tk/MotrFi+rveu
// SIG // 2UQNCLfCd9YaQ3DRbgPeUpLEEAhx2boiVfIfvO2bnTvi
// SIG // Xh1Mg/+XD3sL51WDTtIN677X7K5uR7mf36XWUbwEVe3/
// SIG // J3BMye0qSxPhsblMD8kB7lVlX2kCeGbLPwIDAQABo4IB
// SIG // xzCCAcMwDAYDVR0TAQH/BAIwADBmBgNVHSAEXzBdMFsG
// SIG // C2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEWF2h0dHBz
// SIG // Oi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBka
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMEAGA1UdHwQ5
// SIG // MDcwNaAzoDGGL2h0dHA6Ly90cy1jcmwud3Muc3ltYW50
// SIG // ZWMuY29tL3NoYTI1Ni10c3MtY2EuY3JsMBYGA1UdJQEB
// SIG // /wQMMAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDB3
// SIG // BggrBgEFBQcBAQRrMGkwKgYIKwYBBQUHMAGGHmh0dHA6
// SIG // Ly90cy1vY3NwLndzLnN5bWFudGVjLmNvbTA7BggrBgEF
// SIG // BQcwAoYvaHR0cDovL3RzLWFpYS53cy5zeW1hbnRlYy5j
// SIG // b20vc2hhMjU2LXRzcy1jYS5jZXIwKAYDVR0RBCEwH6Qd
// SIG // MBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTUwHQYD
// SIG // VR0OBBYEFAm1wf6WcpcpQ5rJ4AK6rvj9L7r2MB8GA1Ud
// SIG // IwQYMBaAFK9j1sqjToVy4Ke8QfMpojh/gHViMA0GCSqG
// SIG // SIb3DQEBCwUAA4IBAQAXswqI6VxaXiBrOwoVsmzFqYoy
// SIG // h9Ox9BxTroW+P5v/17y3lIW0x1J+lOi97WGy1KeZ5MPJ
// SIG // k8E1PQvoaApdVpi9sSI70UR617/wbVEyitUj3zgBN/bi
// SIG // Uyt6KxGPt01sejMDG3xrCZQXu+TbWNQhE2Xn7NElyix1
// SIG // mpx//Mm7KmirxH20z6PJbKfZxACciQp3kfRNovsxO4Zu
// SIG // 9uYfUAOGm7/LQqvmdptyWhEBisbvpW+V592uuuYiZfAY
// SIG // WRsRyc2At9iXRx9CCPiscR+wRlOz1LLVo6tQdUgSF4Kt
// SIG // z+BBTzJ+zZUcv5GKCD2kp2cClt8kTKXQQcCCYKOKFzJL
// SIG // 07zPpLSMMYICWjCCAlYCAQEwgYswdzELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MSgwJgYDVQQDEx9TeW1hbnRlYyBTSEEyNTYgVGltZVN0
// SIG // YW1waW5nIENBAhBUWPKq10HWRLyEqXugllLmMAsGCWCG
// SIG // SAFlAwQCAaCBpDAaBgkqhkiG9w0BCQMxDQYLKoZIhvcN
// SIG // AQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTE4MDExNjAzMDMz
// SIG // MFowLwYJKoZIhvcNAQkEMSIEIGNvq6wWGakD/VZJMuld
// SIG // hEA3cImPzWx1bMuPfa/wO90EMDcGCyqGSIb3DQEJEAIv
// SIG // MSgwJjAkMCIEIM96wXrQR+zV/cNoIgMbEtTvB4tvK0xe
// SIG // a6Qfj/LPS61nMAsGCSqGSIb3DQEBAQSCAQB3kGeZHVPW
// SIG // J86TcsBDO9UhjgUtQr0c5zWbmmO/Y890/ZFrYm8aBp3J
// SIG // MNKz3ExtFJF0ymsJsWm0Y/RDcEjlz+v5bjvKH+/2WTSL
// SIG // B8P/W0/sUBdjTAiS23PtqZ1d2sX/fXUHyIiGeyhUK27w
// SIG // ++GdXBX9Xjg3R1aIFZF2yM5kYnLvYNZr79YI2wQB/sLN
// SIG // rlaRRx7lEnF4zWtP3l7Y/RFjw5OstWah9Zsx3tMiRs6k
// SIG // y7uBclLYxl9bxC75FecovhAwxb75lacri5/wSsO1UL+R
// SIG // ytRP8p2q6Tx7IlEDCx027xpOsiRVKqHOdZrobpSOeMAq
// SIG // wH/zQLLUlpeO9m45rNVD92sJ
// SIG // End signature block
