// Main app view.
(function($, UTILS) {

    $.widget('adsk.appview', {
        options: {
            title: 'Book Title',

            minNavigationWidth: 320,

            ui: {
                tabs: {
                    order: ['C', 'I', 'S', 'F'],
                    tocEnabled: true,
                    indexEnabled: true,
                    favoritesEnabled: true,
                    searchEnabled: true
                },

                tools: {
                    order: ['home', 'previous', 'parent', 'following', 'separator',
                            'favorites', 'share', 'separator', 'print'],
                    home: true,
                    previous: true,
                    parent: true,
                    following: true,
                    favorites: true,
                    share: true,
                    print: true
                },

                responsive: true,

				textDirection: 'rtl'
            },

            features: {
                tabsbarParent: 'banner',
                banner: true,
                header: true,
                navigation: true,
                splitter: true,
                breadcrums: true,
                splitter: true
            },

            labels: {
                tabs: {
                }
            }
        },

        /* Private members. */

        _$banner:       undefined,
        _$main:         undefined,
        _$breadcrumbs:  undefined,
        _$splitter:     undefined,
        _$navigation:   undefined,
        _$topic:        undefined,

        _splitterTracking:                  false,
        _splitterMoved:                     false,
        _splitterOriginalX:                 undefined,
        _navigationOriginalWidth:           undefined,
        _topicOriginalWidth:                undefined,
        _navigationOpenedWidth:             undefined,
        _topicOriginalMargin: 	            undefined,
		_dirSign: 							1,
		_dirTopicMarginProperty: 			'margin-left',

        _create: function() {
            console.log('appview: _create');

            var self = this;

            // Show "print" button only when functionality supported.
            if (!window.print) {
                this.options.ui.tools.print = false;
            }

			// Determine text direction.
			if ($('html').attr('dir') == 'rtl') {
				this.options.ui.textDirection = 'rtl';
				this._dirSign = -1;
				this._dirTopicMarginProperty = 'margin-right';
			} else {
				this.options.ui.textDirection = 'ltr';
				this._dirSign = 1;
				this._dirTopicMarginProperty = 'margin-left';
			}

            // Create layout.
            this._render();

            // Initial resize.
            this._resizeHandler();
            this._resizeTabButtons();

            // Perform transition.
            this._$main.transition();

            // Add event handlers.
            this._addEventHandlers(self);

            // Initial app resize.
            this._resizeHandler();

            // Show UI.
            this.element.removeClass('hidden');
        },

        _addEventHandlers: function(self) {
            // Add app resize handler.
            $(window).bind('resize', function() {
                self._resizeHandler();
            });

            self._addViewEventHandlers(self);
        },

        _addViewEventHandlers: function(self) {
            console.log('appview: _addEventHandlers');

            // IE7 hover/active fallback.
            if ($('html').hasClass('ie7') || $('html').hasClass('ie8')) {
                $('a')
                .mousedown(function() {
                    $(this).addClass('active');
                })
                .mouseup(function() {
                    $(this).removeClass('active');
                })
                .mouseenter(function() {
                    $(this).addClass('hover');
                })
                .mouseleave(function() {
                    $(this).removeClass('hover');
                });
            }

            // Add toolbar buttons tab bar links handlers.
            self._on('.button', {
                click: "_handleToolbarButtonClick"
            });

            // Add tab bar links handlers.
            self._on('.tab-button', {
                click: "_handleTabButtonClick"
            });

            // Splitter event handlers.
            self._on(this._$splitter, {
                mousedown: "_splitterMouseDownHandler",
                mouseup: function() {
                    // Toggle pane.
                    if (this._splitterMoved == false) {
                        this._toggleNavigationPane();
                    }
                }
            });

            self._on({
                mousemove: "_splitterMouseMoveHandler",
                mouseup: function() {
                    this._splitterTracking = false;
                    this.element.removeClass('no-user-select');
                }
            });
        },

        _toggleNavigationPane: function() {
            var navigationPaneWidth = this._$navigation.width();

            if (navigationPaneWidth > 0) {
                // Remember current width.
                this._navigationOpenedWidth = navigationPaneWidth;

                // Collapse navigation pane.
                var newTopicMargin = this._topicOriginalMargin - navigationPaneWidth;

                // Resize panes.
                this._resizeSplitPanes(0, 0, 0, 0, newTopicMargin);

                // Hide navigation pane.
                this._$navigation.css('visibility', 'hidden');
            } else {
                // Show navigation pane.
                this._$navigation.css('visibility', 'visible');

                // Open navigation pane.
                var newNavigationPaneWidth = this._navigationOpenedWidth,
                    newTopicMargin = this._topicOriginalMargin + newNavigationPaneWidth;

                this._resizeSplitPanes(0, 0, newNavigationPaneWidth, 0, newTopicMargin);
            }
        },

        _splitterMouseDownHandler: function(event) {
            this._splitterMoved = false;

            // Set up splitter.
            this._splitterOriginalX = event.clientX;
            this._navigationOriginalWidth = this._$navigation.width();
            this._topicOriginalWidth = this._$topic.width();
            this._topicOriginalMargin = parseInt(this._$topic.css(this._dirTopicMarginProperty), 10);

            // Prevent user select.
            this.element.addClass('no-user-select');

            // Acgtivate splitter tracking if pane is not collapsed.
            if (this._navigationOriginalWidth > 0) {
                this._splitterTracking = true;
            }
        },

        _splitterMouseMoveHandler: function(event) {
            this._splitterMoved = true;

            if (this._splitterTracking) {
                // Track splitter.
                var topicWidth = this._$topic.width(),
                    navigationWidth = this._$navigation.width(),
                    deltaX = this._dirSign * (event.clientX - this._splitterOriginalX),
                    minNavigationWidth = this.options.minNavigationWidth;

                if ((deltaX < 0 && navigationWidth > this.options.minNavigationWidth) ||
                    (deltaX > 0 && topicWidth > 0)) {
                    // Resize panes.
                    var newNavigationWidth = this._navigationOriginalWidth + deltaX,
                        newTopicMargin = this._topicOriginalMargin + deltaX,
                        newTopicWidth = this._topicOriginalWidth - deltaX;

                    this._resizeSplitPanes(minNavigationWidth, 0, newNavigationWidth, newTopicWidth, newTopicMargin);

                }
            }
        },

        _resizeSplitPanes: function(minNavigationWidth, minTopicWidth, newNavigationWidth, newTopicWidth, newTopicMargin) {
            if (newNavigationWidth < minNavigationWidth) {
                var delta = minNavigationWidth - newNavigationWidth;
                newNavigationWidth += delta;
                newTopicMargin += delta;
            } else if (newTopicWidth < minTopicWidth) {
                var delta = -(minTopicWidth - newTopicWidth);
                newNavigationWidth += delta;
                newTopicMargin += delta;
            }

            this._$navigation.width(newNavigationWidth);
            this._$topic.css(this._dirTopicMarginProperty, newTopicMargin + 'px');

            this._resizeTabButtons();
        },

        _resizeTabButtons: function() {
            // Stretch tab buttons for non-responsive UI.
            var tabButtons = this._$navigation.find('.tab-button'),
                tabButtonsLen = tabButtons.length;

            // Resize tab buttons if all of them are within navigation pane (non-responsive UI).
            if (tabButtonsLen) {
                var navWidth = this._$navigation.width();
                tabButtons.css('width', Math.floor(navWidth / tabButtons.length)  + 'px');

                // Show tab icons navigation pane is wide enough.
                if (navWidth >= 600) {
                    this._$navigation.addClass('wide');
                } else {
                    this._$navigation.removeClass('wide');
                }
            }
        },

        _render: function() {
            console.log('appview: _render');

            // Create app view markup.
            this.element
            .html(this._renderBanner() +
                  '<div id="main">' +
                    this._renderNavigation() +
                    this._renderSplitter() +
                    '<div id="topic"></div>' +
                  '</div>' +
                  this._renderBreadcrumbs() +
                  '<div id="share" class="hidden"></div>');

            var e = this.element;
            this._$banner       = e.find('#banner');
            this._$main         = e.find('#main');
            this._$splitter     = e.find('#splitter');
            this._$navigation   = e.find('#navigation');
            this._$tabsbar   = this._$navigation.find('.tabsbar');
            this._$topic        = e.find('#topic');
            this._$breadcrumbs  = e.find('#breadcrumbs');
        },

        _renderBanner: function() {
            console.log('appview: _renderBanner');

            var html = ''

            if (this.options.features.banner) {
                html +=  '<div id="banner">' +
                            this._renderHeader() +
                            this._renderToolbar();

                if (this.options.features.tabsbarParent == 'banner') {
                    html += this._renderTabsBar();
                }

                html += '</div>';
            }

            return html;
        },

        _renderToolbar: function() {
            var html = '<div class="toolbar"><div class="wrapper">',
                tabs = this.options.ui.tabs,
                tools = this.options.ui.tools;

            $.each(tools.order, function(index, tool) {
                switch(tool) {
                    case 'separator':
                        html += '<span class="separator"></span>';
                        break;
                    case 'home':
                        if (tools.home) {
                            html += '<a href="#" id="main-home" class="button icon home"></a>';
                        }
                        break;
                    case 'previous':
                        if (tools.previous) {
                            html += '<a href="#" id="main-previous" class="button icon previous"></a>';
                        }
                        break;
                    case 'parent':
                        if (tools.parent) {
                            html += '<a href="#" id="main-parent" class="button icon parent"></a>';
                        }
                        break;
                    case 'following':
                        if (tools.following) {
                            html += '<a href="#" id="main-following" class="button icon following"></a>';
                        }
                        break;
                    case 'favorites':
                        if (tabs.favoritesEnabled && tools.favorites) {
                            html += '<a href="#" id="main-favorites" class="button icon favorites"></a>';
                        }
                        break;
                    case 'share':
                        if (tools.share) {
                            html += '<a href="#" id="main-share" class="button icon share"></a>';
                        }
                        break;
                    case 'print':
                        if (tools.print) {
                            html += '<a href="#" id="main-print" class="button icon print"></a>';
                        }
                        break;
                }
            });

            html += '</div></div>';

            return html;
        },

        _renderTabsBar: function() {
            var that = this,
                html = '<div class="tabsbar"><div class="wrapper">',
                tabs = this.options.ui.tabs,
                caps = this.options.labels.tabs,
                pos = 0;

            $.each(tabs.order, function(index, tab) {
                switch(tab) {
                    case 'C':
                        if (tabs.tocEnabled) {
                            html += that._renderTabsBarButton('tab-button-toc', caps.contents, pos);
                            ++pos;
                        }
                        break;
                    case 'I':
                        if (tabs.indexEnabled) {
                            html += that._renderTabsBarButton('tab-button-index', caps.index, pos);
                            ++pos;
                        }
                        break;
                    case 'S':
                        if (tabs.searchEnabled) {
                            html += that._renderTabsBarButton('tab-button-search', caps.search, pos);
                            ++pos;
                        }
                        break;
                    case 'F':
                        if (tabs.favoritesEnabled) {
                            html += that._renderTabsBarButton('tab-button-favorites', caps.favorites, pos);
                            ++pos;
                        }
                        break;
                }
            });

            html += '</div></div>';

            return html;
        },

        _renderTabsBarButton: function (id, label, pos) {
            var html = '',
                classes = 'tab-button';

            if (pos == 0) {
                classes += ' opened';
            }

            html += '<a id="' + id + '" class="' + classes + '" href="#">' +
                        '<span>' + label + '</span>' +
                    '</a>';

            return html;
        },

        _renderNavigation: function() {
            console.log('appview: _renderNavigation');

            var html = '',
                position = 0,
                that = this,
                features = this.options.features,
                tabs = this.options.ui.tabs;

            if (features.navigation) {
                html += '<div id="navigation">';

                if (this.options.features.tabsbarParent == 'navigation') {
                    html += this._renderTabsBar();
                }

                $.each(tabs.order, function(index, tab) {
                    switch(tab) {
                        case 'C':
                            if (tabs.tocEnabled) {
                                html += that._renderNavigationTab('tab-toc', position);
                                ++position;
                            }
                            break;
                        case 'I':
                            if (tabs.indexEnabled) {
                                html += that._renderNavigationTab('tab-index', position);
                                ++position;
                            }
                            break;
                        case 'S':
                            if (tabs.searchEnabled) {
                                html += that._renderNavigationTab('tab-search', position);
                                ++position;
                            }
                            break;
                        case 'F':
                            if (tabs.favoritesEnabled) {
                                html += that._renderNavigationTab('tab-favorites', position);
                                ++position;
                            }
                            break;
                    }
                });

                html += '</div>';

            } else {
                this.element.addClass('no-navigation');
            }
            return html;
        },

        _renderNavigationTab: function(id, position) {
            var html = '<div id="' + id + '" class = "tab';

            // Only first tab should be visible.
            if (position > 0) {
                html += ' hidden'
            }

            html += '"></div>';

            return html;
        },

        _renderSplitter: function() {
            console.log('appview: _renderSplitter');

            if (this.options.features.navigation) {
                return '<div id="splitter"></div>';
            } else {
                return '';
            }
        },

        _renderBreadcrumbs: function() {
            console.log('appview: _renderBreadcrumbs');

            if (this.options.features.breadcrumbs) {
                return '<div id="breadcrumbs"></div>';
            } else {
                return '';
            }
        },

        _renderHeader: function() {
            console.log('appview: _renderHeader');

            if (this.options.features.header) {
                return  '<div class="header">' +
                            '<div class="wrapper">' +
                                '<a class="title" href="#">' +
                                    '<img class="product-title" src="images/product-title.png"/>' +
                                    '<span>' + this.options.title + '</span>' +
                                '</a>' +
                            '</div>' +
                        '</div>';
            } else {
                return '';
            }
        },

        _handleToolbarButtonClick: function(event) {
            console.log('appview: _handleToolbarButtonClick');

            event.preventDefault();
            event.stopPropagation();

            var $target = $(event.target);

            if (!$target.hasClass('disabled') && $target.attr('id').indexOf('main') != -1) {
                this._trigger('buttonclick', event, $target.attr('id'));
            }
        },

        _handleTabButtonClick: function(event) {
            console.log('appview: _handleTabButtonClick');

            event.preventDefault();
            event.stopPropagation();

            var $target = $(event.currentTarget);

            if (!$target.hasClass('disabled')) {
                this._trigger('tabbuttonclick', event, $target.attr('id'));
            }
        },

        _showTabButtonById: function(id) {
            var tabButtons = this._$navigation.find('.tab-button'),
                targetButton = tabButtons.filter(id);

            if (tabButtons.length) {
                tabButtons.removeClass('opened');
                targetButton.addClass('opened');
            }
        },

        _getShareButtonOffset: function() {
            console.log('appview: _getShareButtonOffset');

        },

        _resizeHandler: function() {
            console.log('appview: _resizeHandler');

            var $window = $(window);

            // Resize according to a new app height.
            var height = $window.height() - this._$banner.height() - this._$breadcrumbs.height();
            this._$main.css('height', height + 'px');

            // Resize according to a new app width.
            if (this.options.ui.responsive) {
                var width = $window.width();
                if (width > 768) {
                    // Full viewport view.
                    $(document.body).removeClass('snap');
                } else {
                    // Snap view.
                    $(document.body).addClass('snap');
                }
            } else {
                this._$navigation.find('.tab').css('height', height - this._$tabsbar.height() + 'px');
            }
        },

        _constructBreadcrumbHtml: function(breadcrumb) {
            console.log('appview: _constructBreadcrumbHtml');

            return '<li>' +
                      '<a href="#!/url=' + breadcrumb.href + '">' +
                        breadcrumb.title +
                      '</a>' +
                   '</li>' +
                   '&rarr;';
        },

        /* Public members. */

        showPreloader: function() {
            console.log('appview: showPreloader');

            $(document.body).addClass('loading');
        },

        hidePreloader: function() {
            console.log('appview: hidePreloader');

            $(document.body).removeClass('loading');
        },

        initGoToPreviousButton: function(enabled) {
            console.log('appview: initGoToPreviousButton');

            if (enabled) {
                this._$banner.find('#main-previous').removeClass('disabled');
            } else {
                this._$banner.find('#main-previous').addClass('disabled');
            }
        },

        initGoToParentButton: function(enabled) {
            console.log('appview: initGoToParentButton');

            if (enabled) {
                this._$banner.find('#main-parent').removeClass('disabled');
            } else {
                this._$banner.find('#main-parent').addClass('disabled');
            }
        },

        initGoToNextButton: function(enabled) {
            console.log('appview: initGoToNextButton');

            if (enabled) {
                this._$banner.find('#main-following').removeClass('disabled');
            } else {
                this._$banner.find('#main-following').addClass('disabled');
            }
        },

        initFavoritesButton: function(enabled) {
            console.log('appview: initFavoritesButton');

            if (enabled) {
                this._$banner.find('#main-favorites').removeClass('disabled');
            } else {
                this._$banner.find('#main-favorites').addClass('disabled');
            }
        },

        getShareButtonOffset: function() {
            console.log('appview: getShareButtonOffset');

            var offset = this.element.find('#main-share').offset();

            return offset;
        },

        showTab: function(id) {
            console.log('appview: showTab');

            var tabSelector = '#tab-' + id,
                tabButtonSelector = '#tab-button-' + id,
                tabToShow = this._$navigation.find(tabSelector),
                tabs = this._$navigation.children('.tab');

            this._showTabButtonById(tabButtonSelector);

            if (tabToShow.length > 0 && tabToShow.hasClass('hidden')) {
                // Hide all tabs.
                tabs.addClass('hidden').attr('style', '');

                // Show selected tab.
                tabToShow.transition().removeClass('hidden');

                // Recalculate tab height.
                this._resizeHandler();

                // Go to current topic in TOC.
                if (tabToShow.attr('id') == 'tab-toc') {
                    //this.expandToc(tabToShow);
                    //this.scrollTocToSelectedTopic(tabToShow);
                }
            } else {
                console.log('appview: error: tab with id ' + id + ' not found.');
            }
        },

        // Flag if topic is in Favorites or not.
        flagFavorites: function(addedToFavorites) {
            console.log('mainview: flagFavorites, added = ', addedToFavorites);

            if (addedToFavorites) {
                this._$banner.find('#main-favorites').addClass('flagged');
            } else {
                this._$banner.find('#main-favorites').removeClass('flagged');
            }
        },

        updateBreadcrumbs: function(breadcrumbs, rootTopicData) {
            var len = breadcrumbs.length,
                i,
                html = '<ol class="breadcrumbs">';

                 + this._constructBreadcrumbHtml({
                    href: rootTopicData.attr.href,
                    title: rootTopicData.data || 'Home'
                });

            for (i = 0; i < len; ++i) {
                html = html + this._constructBreadcrumbHtml(breadcrumbs[i]);
            }

            html += '</ol>';

            this._$breadcrumbs.empty();

            $(html).appendTo(this._$breadcrumbs);
        }
    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIbQQYJKoZIhvcNAQcCoIIbMjCCGy4CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // 2pGPazIC4uzh47KMxezcQWoQ2JXjRYOJZuHe/O1LPFmg
// SIG // ggoWMIIExzCCA6+gAwIBAgIQANMDPwr2vEKYUg3AWz9B
// SIG // yzANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNzA2MjAwMDAw
// SIG // MDBaFw0xODA2MjAyMzU5NTlaMHgxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIDApDYWxpZm9ybmlhMRMwEQYDVQQHDApT
// SIG // YW4gUmFmYWVsMRcwFQYDVQQKDA5BdXRvZGVzaywgSW5j
// SIG // LjENMAsGA1UECwwESVNSQzEXMBUGA1UEAwwOQXV0b2Rl
// SIG // c2ssIEluYy4wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQDEm0ybXlDjakGjfPJQOz8JJyQPZ4frwCIk
// SIG // YuGOWfv+3NZI9Snuy2rQ00G3NuUtmIl0Y0MeypzK5A8s
// SIG // nDGR+bkcRvGn2OmMHS50aggKjsao+rCNMiC8btSwxojg
// SIG // ekzX093J9+/kauZtUESeku1KV2xsBhQ+KTc5TliNgJDJ
// SIG // kB8VW+i2bK0oVx4aMPjX8pivLnbZDYo48U5lQNJtKVBC
// SIG // tnfv0wz2QfRAvXARUxGAnn7xsdgQzYzV6wn+i6N/fMDl
// SIG // e+qWfumXUFSnueYpP85Yi3oNu5kFydgeu34/TTkF1+gD
// SIG // OXfEKLP3cInMUgL7EGTIzHlXaumXFVZobx75J77kq721
// SIG // AgMBAAGjggE+MIIBOjAJBgNVHRMEAjAAMA4GA1UdDwEB
// SIG // /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzBhBgNV
// SIG // HSAEWjBYMFYGBmeBDAEEATBMMCMGCCsGAQUFBwIBFhdo
// SIG // dHRwczovL2Quc3ltY2IuY29tL2NwczAlBggrBgEFBQcC
// SIG // AjAZDBdodHRwczovL2Quc3ltY2IuY29tL3JwYTAfBgNV
// SIG // HSMEGDAWgBTUwAYiSes5S92T4lyhuEd2CXIDWDArBgNV
// SIG // HR8EJDAiMCCgHqAchhpodHRwOi8vcmIuc3ltY2IuY29t
// SIG // L3JiLmNybDBXBggrBgEFBQcBAQRLMEkwHwYIKwYBBQUH
// SIG // MAGGE2h0dHA6Ly9yYi5zeW1jZC5jb20wJgYIKwYBBQUH
// SIG // MAKGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3J0MA0G
// SIG // CSqGSIb3DQEBCwUAA4IBAQApMpNskUmvqk2ygWaCDz+3
// SIG // PTZOZl8Q1FKIoGnv2+iq6wblFUCZt+KgKKyoZoHX6GMd
// SIG // Z/YKeTi5PO4QpKQBgm82kOtKVOdUnRXhUderqaAGrX8W
// SIG // qmXjggda1kRud5e+mdcql+RV1A+RVjjVmSATbeXCC0H5
// SIG // mfvO4o5lnBOfkOQeqKxRjWdEhklOszd+1pQVFX761VyA
// SIG // GfXgr0Jtm9dptQBU1TPybl3+ypIV00IldfMGj1VOKZtd
// SIG // ZrgMiXrdaIWOqRhSEyDdQWvqGzTZm7x5elhF3vcGkRZb
// SIG // 1XqtS7OgDJ7wW0z28NMLA9yfFsPFfOQfMBEEgoCc01P9
// SIG // MBPyiz1XO28kMIIFRzCCBC+gAwIBAgIQfBs1NUrn23Tn
// SIG // QV8RacprqDANBgkqhkiG9w0BAQsFADCBvTELMAkGA1UE
// SIG // BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8w
// SIG // HQYDVQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTow
// SIG // OAYDVQQLEzEoYykgMjAwOCBWZXJpU2lnbiwgSW5jLiAt
// SIG // IEZvciBhdXRob3JpemVkIHVzZSBvbmx5MTgwNgYDVQQD
// SIG // Ey9WZXJpU2lnbiBVbml2ZXJzYWwgUm9vdCBDZXJ0aWZp
// SIG // Y2F0aW9uIEF1dGhvcml0eTAeFw0xNDA3MjIwMDAwMDBa
// SIG // Fw0yNDA3MjEyMzU5NTlaMIGEMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxNTAz
// SIG // BgNVBAMTLFN5bWFudGVjIENsYXNzIDMgU0hBMjU2IENv
// SIG // ZGUgU2lnbmluZyBDQSAtIEcyMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEA15VD1NzfZ645+1KktiYx
// SIG // BHDpt45bKro3aTWVj7vAMOeG2HO73+vRdj+KVo7rLUvw
// SIG // VxhOsY2lM9MLdSPVankn3aPT9w6HZbXerRzx9TW0IlGv
// SIG // IqHBXUuQf8BZTqudeakC1x5JsTtNh/7CeKu/71KunK8I
// SIG // 2TnlmlE+aV8wEE5xY2xY4fAgMxsPdL5byxLh24zEgJRy
// SIG // u/ZFmp7BJQv7oxye2KYJcHHswEdMj33D3hnOPu4Eco4X
// SIG // 0//wsgUyGUzTsByf/qV4IEJwQbAmjG8AyDoAEUF6QbCn
// SIG // ipEEoJl49He082Aq5mxQBLcUYP8NUfSoi4T+IdpcXn31
// SIG // KXlPsER0b21y/wIDAQABo4IBeDCCAXQwLgYIKwYBBQUH
// SIG // AQEEIjAgMB4GCCsGAQUFBzABhhJodHRwOi8vcy5zeW1j
// SIG // ZC5jb20wEgYDVR0TAQH/BAgwBgEB/wIBADBmBgNVHSAE
// SIG // XzBdMFsGC2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEW
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUF
// SIG // BwICMBkaF2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMDYG
// SIG // A1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9zLnN5bWNiLmNv
// SIG // bS91bml2ZXJzYWwtcm9vdC5jcmwwEwYDVR0lBAwwCgYI
// SIG // KwYBBQUHAwMwDgYDVR0PAQH/BAQDAgEGMCkGA1UdEQQi
// SIG // MCCkHjAcMRowGAYDVQQDExFTeW1hbnRlY1BLSS0xLTcy
// SIG // NDAdBgNVHQ4EFgQU1MAGIknrOUvdk+JcobhHdglyA1gw
// SIG // HwYDVR0jBBgwFoAUtnf6aUhHn1MS1cLqBzJ2B9GXBxkw
// SIG // DQYJKoZIhvcNAQELBQADggEBAH/ryqfqi3ZC6z6OIFQw
// SIG // 47e53PpIPhbHD0WVEM0nhqNm8wLtcfiqwlWXkXCD+VJ+
// SIG // Umk8yfHglEaAGLuh1KRWpvMdAJHVhvNIh+DLxDRoIF60
// SIG // y/kF7ZyvcFMnueg+flGgaXGL3FHtgDolMp9Er25DKNMh
// SIG // dbuX2IuLjP6pBEYEhfcVnEsRjcQsF/7Vbn+a4laS8Zaz
// SIG // rS359N/aiZnOsjhEwPdHe8olufoqaDObUHLeqJ/UzSwL
// SIG // NL2LMHhA4I2OJxuQbxq+CBWBXesv4lHnUR7JeCnnHmW/
// SIG // OO8BSgEJJA4WxBR5wUE3NNA9kVKUneFo7wjw4mmcZ26Q
// SIG // CxqTcdQmAsPAWiMxghCDMIIQfwIBATCBmTCBhDELMAkG
// SIG // A1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBv
// SIG // cmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBO
// SIG // ZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAz
// SIG // IFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMgIQANMD
// SIG // Pwr2vEKYUg3AWz9ByzANBglghkgBZQMEAgEFAKB8MBAG
// SIG // CisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgor
// SIG // BgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEE
// SIG // AYI3AgEVMC8GCSqGSIb3DQEJBDEiBCCBEvBt/xrsfEsp
// SIG // SGe4lwlxDr2I5421AceQqnCgjAKIxzANBgkqhkiG9w0B
// SIG // AQEFAASCAQA5GGhJtosoxyOwwlO8EOdqlpUPj5eedZR/
// SIG // Gpg4trZ2jTSIkFIjSv8jXOBhq/VG7Y+KUxk1qtZ1YzaZ
// SIG // wM62NLQ540/OPV4B3btzi/TRPET/L2y9jHrHIFO8ftQE
// SIG // o5uuzoTd8FURjimrlU73si8dkow90SCdgTw83D2BvgaZ
// SIG // XL++UyRFTdD52xeFArL1NLyQs6/b5F+/E42kGuQqI0U7
// SIG // pki1ACcHLKW1z53A69alKFSZ744+owERCV2GXCPd0ttn
// SIG // WvaxkZkVSNzcbquEFdYvP2IDIfCmWLQo8/9ZZKMSg4hd
// SIG // DILBKodz8/xMnTVtJ76siWxaVzjiBXZcNygIgNUr5eEy
// SIG // oYIOPDCCDjgGCisGAQQBgjcDAwExgg4oMIIOJAYJKoZI
// SIG // hvcNAQcCoIIOFTCCDhECAQMxDTALBglghkgBZQMEAgEw
// SIG // ggEOBgsqhkiG9w0BCRABBKCB/gSB+zCB+AIBAQYLYIZI
// SIG // AYb4RQEHFwMwMTANBglghkgBZQMEAgEFAAQgUvv2fpEk
// SIG // k3j5tSohkFFhWIzAihvj3ZEeod5oJWt01ecCFDZ9Bq32
// SIG // 1nBEjm4wguHaidJCRIpdGA8yMDE4MDExNjAzMDMzNFow
// SIG // AwIBHqCBhqSBgzCBgDELMAkGA1UEBhMCVVMxHTAbBgNV
// SIG // BAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQL
// SIG // ExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTEwLwYDVQQD
// SIG // EyhTeW1hbnRlYyBTSEEyNTYgVGltZVN0YW1waW5nIFNp
// SIG // Z25lciAtIEcyoIIKizCCBTgwggQgoAMCAQICEHsFsdRJ
// SIG // aFFE98mJ0pwZnRIwDQYJKoZIhvcNAQELBQAwgb0xCzAJ
// SIG // BgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5j
// SIG // LjEfMB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29y
// SIG // azE6MDgGA1UECxMxKGMpIDIwMDggVmVyaVNpZ24sIElu
// SIG // Yy4gLSBGb3IgYXV0aG9yaXplZCB1c2Ugb25seTE4MDYG
// SIG // A1UEAxMvVmVyaVNpZ24gVW5pdmVyc2FsIFJvb3QgQ2Vy
// SIG // dGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMTYwMTEyMDAw
// SIG // MDAwWhcNMzEwMTExMjM1OTU5WjB3MQswCQYDVQQGEwJV
// SIG // UzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24x
// SIG // HzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsx
// SIG // KDAmBgNVBAMTH1N5bWFudGVjIFNIQTI1NiBUaW1lU3Rh
// SIG // bXBpbmcgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQC7WZ1ZVU+djHJdGoGi61XzsAGtPHGsMo8F
// SIG // a4aaJwAyl2pNyWQUSym7wtkpuS7sY7Phzz8LVpD4Yht+
// SIG // 66YH4t5/Xm1AONSRBudBfHkcy8utG7/YlZHz8O5s+K2W
// SIG // OS5/wSe4eDnFhKXt7a+Hjs6Nx23q0pi1Oh8eOZ3D9Jqo
// SIG // 9IThxNF8ccYGKbQ/5IMNJsN7CD5N+Qq3M0n/yjvU9bKb
// SIG // S+GImRr1wOkzFNbfx4Dbke7+vJJXcnf0zajM/gn1kze+
// SIG // lYhqxdz0sUvUzugJkV+1hHk1inisGTKPI8EyQRtZDqk+
// SIG // scz51ivvt9jk1R1tETqS9pPJnONI7rtTDtQ2l4Z4xaE3
// SIG // AgMBAAGjggF3MIIBczAOBgNVHQ8BAf8EBAMCAQYwEgYD
// SIG // VR0TAQH/BAgwBgEB/wIBADBmBgNVHSAEXzBdMFsGC2CG
// SIG // SAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8v
// SIG // ZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkaF2h0
// SIG // dHBzOi8vZC5zeW1jYi5jb20vcnBhMC4GCCsGAQUFBwEB
// SIG // BCIwIDAeBggrBgEFBQcwAYYSaHR0cDovL3Muc3ltY2Qu
// SIG // Y29tMDYGA1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9zLnN5
// SIG // bWNiLmNvbS91bml2ZXJzYWwtcm9vdC5jcmwwEwYDVR0l
// SIG // BAwwCgYIKwYBBQUHAwgwKAYDVR0RBCEwH6QdMBsxGTAX
// SIG // BgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTMwHQYDVR0OBBYE
// SIG // FK9j1sqjToVy4Ke8QfMpojh/gHViMB8GA1UdIwQYMBaA
// SIG // FLZ3+mlIR59TEtXC6gcydgfRlwcZMA0GCSqGSIb3DQEB
// SIG // CwUAA4IBAQB16rAt1TQZXDJF/g7h1E+meMFv1+rd3E/z
// SIG // ociBiPenjxXmQCmt5l30otlWZIRxMCrdHmEXZiBWBpgZ
// SIG // jV1x8viXvAn9HJFHyeLojQP7zJAv1gpsTjPs1rSTyEyQ
// SIG // Y0g5QCHE3dZuiZg8tZiX6KkGtwnJj1NXQZAv4R5NTtzK
// SIG // EHhsQm7wtsX4YVxS9U72a433Snq+8839A9fZ9gOoD+NT
// SIG // 9wp17MZ1LqpmhQSZt/gGV+HGDvbor9rsmxgfqrnjOgC/
// SIG // zoqUywHbnsc4uw9Sq9HjlANgCk2g/idtFDL8P5dA4b+Z
// SIG // idvkORS92uTTw+orWrOVWFUEfcea7CMDjYUq0v+uqWGB
// SIG // MIIFSzCCBDOgAwIBAgIQVFjyqtdB1kS8hKl7oJZS5jAN
// SIG // BgkqhkiG9w0BAQsFADB3MQswCQYDVQQGEwJVUzEdMBsG
// SIG // A1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNV
// SIG // BAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxKDAmBgNV
// SIG // BAMTH1N5bWFudGVjIFNIQTI1NiBUaW1lU3RhbXBpbmcg
// SIG // Q0EwHhcNMTcwMTAyMDAwMDAwWhcNMjgwNDAxMjM1OTU5
// SIG // WjCBgDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFu
// SIG // dGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRl
// SIG // YyBUcnVzdCBOZXR3b3JrMTEwLwYDVQQDEyhTeW1hbnRl
// SIG // YyBTSEEyNTYgVGltZVN0YW1waW5nIFNpZ25lciAtIEcy
// SIG // MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
// SIG // mfP82AQJA4b511ymk8BCfOp8Y89dAOKO88CQ348p9Rjq
// SIG // lLeS5dewoHOB6OkKm0p8Af+dj6Q5pw7qRfQiDDpw7TlF
// SIG // i+TFG1zwRWhGJAVjdpsc/J5sKrFW5Yp/UnGu8jXVRiMG
// SIG // HM9ILR20zbjZdiOOHP8+v7sGXGkHpmUO+F6ufS7tTa41
// SIG // 78nXAEL9KJUOn11yQgm8w9pE0u3MR4Tk/MotrFi+rveu
// SIG // 2UQNCLfCd9YaQ3DRbgPeUpLEEAhx2boiVfIfvO2bnTvi
// SIG // Xh1Mg/+XD3sL51WDTtIN677X7K5uR7mf36XWUbwEVe3/
// SIG // J3BMye0qSxPhsblMD8kB7lVlX2kCeGbLPwIDAQABo4IB
// SIG // xzCCAcMwDAYDVR0TAQH/BAIwADBmBgNVHSAEXzBdMFsG
// SIG // C2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEWF2h0dHBz
// SIG // Oi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBka
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMEAGA1UdHwQ5
// SIG // MDcwNaAzoDGGL2h0dHA6Ly90cy1jcmwud3Muc3ltYW50
// SIG // ZWMuY29tL3NoYTI1Ni10c3MtY2EuY3JsMBYGA1UdJQEB
// SIG // /wQMMAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDB3
// SIG // BggrBgEFBQcBAQRrMGkwKgYIKwYBBQUHMAGGHmh0dHA6
// SIG // Ly90cy1vY3NwLndzLnN5bWFudGVjLmNvbTA7BggrBgEF
// SIG // BQcwAoYvaHR0cDovL3RzLWFpYS53cy5zeW1hbnRlYy5j
// SIG // b20vc2hhMjU2LXRzcy1jYS5jZXIwKAYDVR0RBCEwH6Qd
// SIG // MBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTUwHQYD
// SIG // VR0OBBYEFAm1wf6WcpcpQ5rJ4AK6rvj9L7r2MB8GA1Ud
// SIG // IwQYMBaAFK9j1sqjToVy4Ke8QfMpojh/gHViMA0GCSqG
// SIG // SIb3DQEBCwUAA4IBAQAXswqI6VxaXiBrOwoVsmzFqYoy
// SIG // h9Ox9BxTroW+P5v/17y3lIW0x1J+lOi97WGy1KeZ5MPJ
// SIG // k8E1PQvoaApdVpi9sSI70UR617/wbVEyitUj3zgBN/bi
// SIG // Uyt6KxGPt01sejMDG3xrCZQXu+TbWNQhE2Xn7NElyix1
// SIG // mpx//Mm7KmirxH20z6PJbKfZxACciQp3kfRNovsxO4Zu
// SIG // 9uYfUAOGm7/LQqvmdptyWhEBisbvpW+V592uuuYiZfAY
// SIG // WRsRyc2At9iXRx9CCPiscR+wRlOz1LLVo6tQdUgSF4Kt
// SIG // z+BBTzJ+zZUcv5GKCD2kp2cClt8kTKXQQcCCYKOKFzJL
// SIG // 07zPpLSMMYICWjCCAlYCAQEwgYswdzELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MSgwJgYDVQQDEx9TeW1hbnRlYyBTSEEyNTYgVGltZVN0
// SIG // YW1waW5nIENBAhBUWPKq10HWRLyEqXugllLmMAsGCWCG
// SIG // SAFlAwQCAaCBpDAaBgkqhkiG9w0BCQMxDQYLKoZIhvcN
// SIG // AQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTE4MDExNjAzMDMz
// SIG // NFowLwYJKoZIhvcNAQkEMSIEIJp2NUsHhQ6lH+2v50w6
// SIG // dccxKOCll0/TLm0BXse9Ui1ZMDcGCyqGSIb3DQEJEAIv
// SIG // MSgwJjAkMCIEIM96wXrQR+zV/cNoIgMbEtTvB4tvK0xe
// SIG // a6Qfj/LPS61nMAsGCSqGSIb3DQEBAQSCAQCJt1EHZoqx
// SIG // Ov5t6F43lSuaxA1IdrenlXY8XmgZOOn2qCn6Ok9wQUnc
// SIG // Ky1OKrsANUC376VJ4MfHKgxgOjBeW9n8lVpXL1My7PPd
// SIG // u7kPRsYHgyeJFB2pt8VgWQSWByBec+fnErxXWXSIRrGb
// SIG // 21QfrS4adweB5T9d/D/k6W9pZ3OobOKXasIyQfPhUga4
// SIG // 7zPue1vDTn1bq9wL8uwuJlYa+nOi6TLGdS8y0pqR5paf
// SIG // en7OJ8K3gHD1Vns+Bub+Bl2b3Wnb9OxqEsQN+MzhWIRK
// SIG // qx3+9z6JwB9fe2imVsMinfRK+mdpu2stH9mCT/m9rTKL
// SIG // JVR1+G0MT+yAUkooIdYcnlym
// SIG // End signature block
